# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour le test des tunnels
"""
from os import system
from os.path import isfile
from IPy import IP
from twisted.internet.utils import getProcessOutput, getProcessOutputAndValue
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.config import ACTIVER_HAUTE_DISPO, SW_THREADS, \
                                               NOM_MACHINE, NOM_MACHINE_MAITRE, \
                                               NOM_MACHINE_ESCLAVE
from zephir.monitor.agentmanager.agent import Agent, RRDAgent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img, log


def parse_ipsec_statusall(response):
    """parses ipsec statusall info
    response format :
        Connections:
        amontestha-sphynxtestha1:  192.168.0.6...192.168.0.16, dpddelay=120s
        amontestha-sphynxtestha1:   local:  [C=fr, O=gouv, OU=education, OU=ac-dijon, CN=0210066H-15] uses public key authentication
        amontestha-sphynxtestha1:   remote: [C=fr, O=gouv, OU=education, OU=ac-dijon, CN=AGRIATES-DIJON-10] uses any authentication
        dmz-reseau172:   child:  10.121.11.0/24 === 172.16.0.0/12 , dpdaction=clear
        admin-reseau_eth1:   child:  10.21.11.0/24 === 172.30.107.0/25 , dpdaction=clear
        admin-reseau10:   child:  10.21.11.0/24 === 10.0.0.0/8 , dpdaction=clear
        admin-reseau172:   child:  10.21.11.0/24 === 172.16.0.0/12 , dpdaction=clear
        Security Associations:
        amontestha-sphynxtestha1[2]: ESTABLISHED 15 minutes ago, 192.168.0.6[C=fr, O=gouv, OU=education, OU=ac-dijon, CN=0210066H-15]...192.168.0.16[C=fr, O=gouv, OU=education, OU=ac-dijon, CN=AGRIATES-DIJON-10]
        amontestha-sphynxtestha1[2]: IKE SPIs: 7c61e6943c503455_i* 670a092581afa023_r, public key reauthentication in 36 minutes
        amontestha-sphynxtestha1[2]: IKE proposal: AES_CBC_128/HMAC_SHA2_256_128/PRF_HMAC_SHA2_256/MODP_2048
        admin-reseau_eth1{2}:  INSTALLED, TUNNEL, ESP SPIs: c5abf3db_i c1abb8d6_o
        admin-reseau_eth1{2}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 15 minutes
        admin-reseau_eth1{2}:   10.21.11.0/24 === 172.30.107.0/25
        admin-reseau10{3}:  INSTALLED, TUNNEL, ESP SPIs: c8e253f3_i c9bbec32_o
        admin-reseau10{3}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 15 minutes
        admin-reseau10{3}:   10.21.11.0/24 === 10.0.0.0/8
        dmz-reseau172{1}:  INSTALLED, TUNNEL, ESP SPIs: c64a5816_i c6b2622d_o
        dmz-reseau172{1}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 13 minutes
        dmz-reseau172{1}:   10.121.11.0/24 === 172.16.0.0/12
        admin-reseau172{4}:  INSTALLED, TUNNEL, ESP SPIs: c1b2d485_i c5be1b6d_o
        admin-reseau172{4}:  AES_GCM_16_128, 0 bytes_i, 0 bytes_o, rekeying in 16 minutes
        admin-reseau172{4}:   10.21.11.0/24 === 172.16.0.0/12
    """
    idle_threads = total = up = down = 0
    try:
        if "statusall" in response:
            idle_threads = total = up = down = -1
            raise Exception("Frozen ipsec")
        elif "NoConfig" in response:
            idle_threads = total = up = down = -2
            raise Exception("No VPN configuration")
        elif response == "":
            idle_threads = total = up = down = -3
            # Ne devrait plus se produire maintenant qe le code retour est traité
            raise Exception("Empty response")
        response = response.split('\n')
        for line in response:
            if "worker threads: " in line:
                idle_threads = int(line.split('worker threads: ')[1].split(' ')[0])
                break
        # On mémorise les connexions définies
        startindex_connections_list = response.index('Connections:')
        endindex_connections_list = 0
        for line in response:
            if 'Security Associations' in line:
                break
            endindex_connections_list += 1
        tunnels = []
        peer_name = None
        for line in response[startindex_connections_list+1:endindex_connections_list]:
            data = line.split(':')
            if 'child' not in line:
                # Il s'agit d'un peer
                tmp_name = data[0].lstrip()
                if tmp_name != peer_name:
                    # Nouveau peer: on extrait le nom et les IP src et dst
                    peer_name = tmp_name
                    subnets = data[1].split('...')
                    src = subnets[0].strip()
                    dst = subnets[1].split(',')[0].strip()
                    dst = dst.split(' ')[0].strip()
                elif 'local' in data[1]:
                    # On extrait le certif source
                    src = src + data[2].split('uses')[0].strip()
                elif 'remote' in data[1]:
                    # On extrait le certif dest et on ajoute à la liste
                    # Le child_name est vide si c'est un peer
                    dst = dst + data[2].split('uses')[0].strip()
                    tunnels.append({'peer_name':peer_name, 'child_name':'',
                                    'src':src, 'dst':dst, 'status':''})
            else:
                # Il s'agit d'un child
                child_name = data[0].lstrip()
                subnets = data[2].split(' === ')
                src = subnets[0].strip()
                if src == '':
                    src = 'Réseau(x) source et/ou destination non renseigné(s) !!!'
                dst = subnets[1].split(',')[0].strip()
                dst = dst.split(' ')[0].strip()
                if dst == '':
                    dst = 'Réseau(x) source et/ou destination non renseigné(s) !!!'
                tunnels.append({'peer_name':peer_name, 'child_name':child_name,
                                'src':src, 'dst':dst, 'status':''})
                total += 1

        # On mémorise les connexions actives
        active_tunnels = {}
        peer_name = None
        child_name = None
        for line in response[endindex_connections_list+1:]:
            if 'none' in line:
                active_tunnels['none'] = {'peer_name':peer_name,
                                          'child_name':child_name,
                                          'src':'', 'dst':'', 'status':''}
            else:
                data = line.split(':')
                if '[' in data[0]:
                    # Il s'agit d'un peer
                    tmp_name = data[0].split('[')[0].lstrip()
                    if tmp_name != peer_name:
                        # Nouveau peer: on extrait le nom, IP et certif src et dest et le statut
                        peer_name = tmp_name
                        if 'ESTABLISHED' in data[1]:
                            subnets = data[1].split('...')
                            src = subnets[0].split('ago,')[1].strip()
                            dst = subnets[1].strip()
                            status = data[1].strip().split(' ')[0]
                        elif 'CONNECTING' in data[1]:
                            src = ''
                            dst = ''
                            status = 'CONNECTING'
                        elif 'DELETING' in data[1]:
                            src = ''
                            dst = ''
                            status = 'DELETING'
                        else:
                            src = ''
                            dst = ''
                            status = 'UNKNOWN'
                        if status != 'CONNECTING' and status != 'DELETING':
                            active_tunnels[peer_name] = {'peer_name':peer_name, 'child_name':'', 'src':src, 'dst':dst, 'status':status}
                    else:
                        if status == 'CONNECTING':
                            if '0000000000000000_r' in data[2]:
                                src = ''
                                dst = ''
                                status = 'NO_REMOTE_KEY'
                                active_tunnels[peer_name] = {'peer_name':peer_name, 'child_name':'', 'src':src, 'dst':dst, 'status':status}
                                log.msg("""No remote key for "{0}" connection""".format(peer_name))
                        elif status == 'DELETING':
                            if 'Tasks queued' in data[1]:
                                log.msg("""DELETING SA still in task queued for "{0}" connection""".format(peer_name))
                        elif 'Tasks queued' in data[1]:
                            if len(data[2].split(' ')) > 10:
                                log.msg("""Too much tasks queued for "{0}" connection""".format(peer_name))
                elif '{' in data[0]:
                    # Il s'agit d'un child
                    tmp_name = data[0].split('{')[0].lstrip()
                    if tmp_name != child_name:
                        # Nouveau child: on extrait le nom et le statut
                        child_name = tmp_name
                        status = data[1].split(',')[0].strip()
                    elif "===" in data[1]:
                        # Dans les lignes suivantes, on extrait les réseaux src et dst
                        subnets = data[1].split('===')
                        src = subnets[0].strip()
                        dst = subnets[1].strip()
                        active_tunnels[peer_name+child_name] = {'peer_name':peer_name, 'child_name':child_name, 'src':src, 'dst':dst, 'status':status}

        # On regarde l'état des tunnels
        if active_tunnels.has_key('none'):
            # Aucune connexion active
            for tunnel in tunnels:
                tunnel['status'] = 'none'
                down += 1
        else:
            for tunnel in tunnels:
                # on renseigne le status de la connexion
                if tunnel['child_name'] == '':
                    try:
                        if active_tunnels.has_key(tunnel['peer_name']):
                            if active_tunnels[tunnel['peer_name']]['status'] == 'ESTABLISHED':
                                tunnel['status'] = 'On'
                            elif active_tunnels[tunnel['peer_name']]['status'] == 'NO_REMOTE_KEY':
                                tunnel['dst'] += " injoignable !!!"
                                tunnel['status'] = 'Off'
                            else:
                                tunnel['status'] = 'Off'
                        else:
                            tunnel['status'] = 'Off'
                    except:
                        tunnel['status'] = 'Off'
                # on renseigne le status de chaque tunnel
                else:
                    try:
                        if active_tunnels.has_key(tunnel['peer_name']+tunnel['child_name']):
                            if active_tunnels[tunnel['peer_name']+tunnel['child_name']]['status'] == 'INSTALLED':
                                tunnel['status'] = 'On'
                                up += 1
                            else:
                                tunnel['status'] = 'Off'
                                down += 1
                        else:
                            tunnel['status'] = 'Off'
                    except:
                        tunnel['status'] = 'Off'
                        down += 1

        # Si on est sur Amon, on vérifie les test-rvp
        ip_list = []
        try:
            test_rvp = open('/usr/share/eole/test-rvp','r').read().split('\n')
            for line in test_rvp:
                if 'fping' in line:
                    ip_list.append(line.split('fping')[1].strip())
        except:
            pass
        try:
            test_rvp = open('/usr/share/eole/test-rvp_more_ip','r').read().split('\n')
            for line in test_rvp:
                if 'fping' in line:
                    ip_list.append(line.split('fping')[1].strip())
        except:
            pass
        for tunnel in tunnels:
            if tunnel['child_name'] != '':
                ip_in_tunnel = ip_err = 0
                try:
                    subnet = IP(tunnel['dst'])
                    for ip in ip_list:
                        if IP(ip) in subnet:
                            ip_in_tunnel +=1
                            cmd = "fping -r 1 -t 2000 "+str(ip)+"  >/dev/null 2>&1"
                            errcode = system(cmd)
                            if errcode != 0:
                                tunnel['dst'] += " -- "+ip+" injoignable"
                                if tunnel['status'] == 'On':
                                    tunnel['status'] = 'Off'
                                    down +=1
                                    up -=1
                            else:
                                tunnel['dst'] += " -- "+ip+" joignable"
                    if tunnel['status'] == 'Off':
                        log.msg("tunnel "+str(tunnel['src'])+" vers "+str(tunnel['dst']+ " en erreur"))
                except:
                    pass

    except Exception, msg:
        tunnels = []
        log.msg(msg)

    return tunnels, total, up, down, idle_threads

def gen_tunnels_status_file(response):
    tunnels, total, up, down, idle_threads = parse_ipsec_statusall(response)
    tunnels_status_content = []
    tunnels_status_content.append(str(tunnels))
    tunnels_status_content.append(str(total))
    tunnels_status_content.append(str(up))
    tunnels_status_content.append(str(down))
    tunnels_status_content.append(str(idle_threads))
    with open('/tmp/tunnels_status.txt', "w") as tunnels_status:
        tunnels_status.write('\n'.join(tunnels_status_content))
        tunnels_status.close()
    return tunnels, total, up, down, idle_threads

class RvpAmon(Agent):

    def __init__(self, name,
                 **params):
        Agent.__init__(self, name, **params)
        self.status = status.Unknown()
        self.retcode = 0
        self.table = TableData([
            ('peer_name', "Connexion", {'align':'left'}, None),
            ('child_name', "Tunnel", {'align':'left'}, None),
            ('status', "Etat", {'align':'center'}, status_to_img),
            ('src', "Source", {'align':'left'}, None),
            ('dst', "Destination", {'align':'left'}, None),
            ])
        self.data = [self.table]
        self.measure_data = {}

    def frozen_ipsec_callback(self, response):
        if 'NoConfig' in response:
            cmd = 'echo'
            param = ['NoConfig']
        elif response != '':
            cmd = 'echo'
            param = ['statusall']
        else:
            cmd = "/usr/sbin/ipsec"
            param = ['statusall']
        update = getProcessOutputAndValue(cmd, param)
        return update.addCallbacks(self.callback_tunnels, self.errback_tunnels)

    def frozen_ipsec_errback(self):
        log.msg("It doesn't work")

    def xfrm_policy_callback(self, ret):
        out, err, self.retcode = ret
        if self.retcode != 0:
            self.status = status.Error("Erreur xfrm policies")
            log.msg("Erreur xfrm policies")
            return self.status

        if isfile('/usr/share/eole/test-rvp'):
            cmd = "pgrep"
            param = ['stroke']
        else:
            cmd = "echo"
            param = ['NoConfig']
        res = getProcessOutput(cmd, param)
        return res.addCallbacks(self.frozen_ipsec_callback, self.frozen_ipsec_errback)

    def xfrm_policy_errback(self):
        log.msg("Erreur xfrm policies")

    def measure(self):
        self.status = status.OK()
        cmd = "/etc/ipsec.d/ip_xfrm_policy"
        res = getProcessOutputAndValue(cmd, ['status'])
        return res.addCallbacks(self.xfrm_policy_callback, self.xfrm_policy_errback)

    def callback_tunnels(self,response):
        out, err, self.retcode = response
        if self.retcode != 0:
            self.status = status.Error()
            self.measure_data = {}
            tunnels = {}
        else:
            tunnels, total, up, down, idle_threads = gen_tunnels_status_file(out)
            # On force l'action en cas d'erreur
            if total == -1 and down == -1 and up == -1 and idle_threads == -1:
                log.msg("Can't parse 'ipsec statusall' response")
                self.last_status = status.OK()
                self.measure_data = {}
                tunnels = {}
                up = down = 0
                self.last_status = status.OK()
                self.status = status.Warn("Mesure non disponible")
            elif total == -2 and down == -2 and up == -2 and idle_threads == -2:
                self.measure_data = {}
                tunnels = {}
                up = down = 0
                self.last_status = status.OK()
                self.status = status.Warn("VPN non configuré")
            elif down > 0 or total != up + down:
                self.status = status.Error("Anomalie dans les tunnels")
            for data in tunnels:
                self.measure_data[data['peer_name']+data['child_name']] = (data['src'], data['dst'], data['status'])
            self.measure_data['ok'] = up
            self.measure_data['bad'] = down
        return {'statistics':tunnels}

    def errback_tunnels(self, err):
        self.status = status.Error("Erreur d'execution : ipsec status")
        return {'statistics':{}}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['statistics']

    def check_status(self):
        return self.status

class RvpSphynx(RRDAgent):

    def __init__(self, name, **params):
        RRDAgent.__init__(self, name, **params)
        self.status = status.Unknown()
        self.retcode = 0
        self.measure_data = {}
        self.up = 0
        self.down = 0
        self.total = 0
        self.pourcentok = 100
        self.localnode = NOM_MACHINE
        if ACTIVER_HAUTE_DISPO == 'maitre':
            self.remotenode = NOM_MACHINE_ESCLAVE
        elif ACTIVER_HAUTE_DISPO == 'esclave':
            self.remotenode = NOM_MACHINE_MAITRE
        else:
            self.remotenode = None

    def set_rvp_status(self, response):
        out, err, self.retcode = response
        if self.retcode != 0:
            self.status = status.Error("strongSwan en erreur")
            self.up = self.down = self.total = self.pourcentok = 0
        else:
            tunnels, self.total, self.up, self.down, self.idle_threads = gen_tunnels_status_file(out)
            self.measure_data['ok'] = self.up
            self.measure_data['bad'] = self.down
            if self.total > 0:
                self.pourcentok = (100*self.up)/self.total
            elif self.total == -1:
                self.pourcentok = -1
            else:
                self.pourcentok = 100
            if self.pourcentok == -1:
                self.status = status.Warn("Mesure non disponible")
            elif self.pourcentok < 75:
                self.status = status.Error("plus de 25% de tunnels en erreur")
            elif self.pourcentok < 95:
                self.status = status.Warn("plus de 5% de tunnels en erreur")

# test frozen
    def frozen_ipsec_callback(self, response):
        if response != '':
            cmd = 'echo'
        else:
            cmd = "/usr/sbin/ipsec"
        update = getProcessOutputAndValue(cmd, ['statusall'])
        return update.addCallbacks(self.callback_tunnels, self.errback_tunnels)

    def frozen_ipsec_errback(self):
        log.msg("It doesn't work")

    def measure(self):
        self.status = status.OK()
        cmd = "pgrep"
        res = getProcessOutput(cmd, ['stroke'])
        return res.addCallbacks(self.frozen_ipsec_callback, self.frozen_ipsec_errback)

    def callback_tunnels(self,response):
        if ACTIVER_HAUTE_DISPO == 'non':
            self.set_rvp_status(response)
        else:
            # Si haute dispo activée, on teste sur quel nœud ipsec est activé
            cmd = "crm resource show VIPCluster 2>>/dev/null| grep -sqE \"VIPCluster is running on: {0} $\"".format(self.localnode)
            errcode = system(cmd)
            # Si le nœud est actif, on renvoi les stats
            if errcode == 0:
                self.set_rvp_status(response)
            else:
                cmd = "crm resource show VIPCluster 2>>/dev/null| grep -sqE \"VIPCluster is running on: {0} $\"".format(self.remotenode)
                errcode = system(cmd)
                if errcode == 0:
                    # Si IPsec tourne sur le nœud distant, on dit que tout est bon
                    log.msg("Haute dispo : nœud inactif")
                    self.up = self.down = self.total = 0
                    self.pourcentok = 100
                else:
                    log.msg("Haute dispo : aucun nœud actif")
                    self.status = status.Error("Haute dispo : aucun nœud actif")
                    self.up = self.down = self.total = self.pourcentok = 0
        return {'ok':self.up, 'bad':self.down, 'total':self.total, 'pourcentok':self.pourcentok}

    def errback_tunnels(self, err):
        self.status = status.Error("Erreur d'execution : ipsec statusall")
        self.up = self.down = self.total = self.pourcentok = 0
        return {'ok':self.up, 'bad':self.down, 'total':self.total, 'pourcentok':self.pourcentok}

    def check_status(self):
        return self.status

class strongSwan_threads(RRDAgent):

    def __init__(self, name, **params):
        RRDAgent.__init__(self, name, **params)
        self.status = status.Unknown()
        self.retcode = 0
        self.measure_data = {}
        self.max_threads = SW_THREADS
        self.localnode = NOM_MACHINE
        if ACTIVER_HAUTE_DISPO == 'maitre':
            self.remotenode = NOM_MACHINE_ESCLAVE
        elif ACTIVER_HAUTE_DISPO == 'esclave':
            self.remotenode = NOM_MACHINE_MAITRE
        else:
            self.remotenode = None
        self.running_threads = 0

    def measure(self):
        self.status = status.OK()
        cmd = "cat"
        res = getProcessOutput(cmd, ['/tmp/tunnels_status.txt'])
        return res.addCallbacks(self.callback_tunnels, self.errback_tunnels)

    def set_threads_status(self,response):
        try:
            measure_array = response.split('\n')
            tunnels = eval(measure_array[0])
            up = int(measure_array[1])
            down = int(measure_array[2])
            running_threads = int(measure_array[3])
            if running_threads == -1:
                self.status = status.Warn("Mesure non disponible")
                self.max_threads = 0
                self.running_threads = 0
            else:
                self.max_threads = SW_THREADS
                self.running_threads = running_threads
                if (self.max_threads - self.running_threads) < 2:
                    self.status = status.Error("Moins de 2 threads strongSwan disponibles")
                elif (self.max_threads - self.running_threads) < 5:
                        self.status = status.Warn("Moins de 5 threads strongSwan disponibles")
        except:
            self.max_threads = 0
            self.running_threads = 0
            self.status = status.Error("Erreur de lecture du fichier des mesures")


    def callback_tunnels(self,response):
        if ACTIVER_HAUTE_DISPO == 'non':
            self.set_threads_status(response)
        else:
            # Si haute dispo activée, on teste sur quel nœud ipsec est activé
            cmd = "crm resource show VIPCluster 2>>/dev/null| grep -sqE \"VIPCluster is running on: {0} $\"".format(self.localnode)
            errcode = system(cmd)
            # Si le nœud est actif, on renvoie les stats
            if errcode == 0:
                log.msg("Haute dispo : nœud actif")
                if self.retcode != 0:
                    self.set_threads_status(response)
            else:
                cmd = "crm resource show VIPCluster 2>>/dev/null| grep -sqE \"VIPCluster is running on: {0} $\"".format(self.remotenode)
                errcode = system(cmd)
                if errcode == 0:
                    # Si IPsec tourne sur le nœud distant, tous les threads sont disponibles
                    self.running_threads = 0
                else:
                    self.status = status.Error("Haute dispo : aucun nœud actif")
                    self.max_threads = 0
                    self.running_threads = 0
        return {'max_threads':self.max_threads,'running_threads':self.running_threads}

    def errback_tunnels(self, err):
        self.status = status.Error("Erreur de lecture du fichier des mesures")
        return {'max_threads':0,'running_threads':0}

    def check_status(self):
        return self.status
