# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2011
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent Systemd
"""
from twisted.internet.utils import getProcessOutputAndValue
from twisted.internet.defer import DeferredList
from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.config import MODE_CONTENEUR_ACTIF, GROUPS
from zephir.monitor.agentmanager.util import status_to_img
import re
import datetime
from dateutil.parser import parse as parse_time_str
from dateutil.tz import tzlocal


SYSTEM_STATE_RE = re.compile(r'^SystemState=(?P<system_state>.*)$', re.M)
USERSPACE_TIMESTAMP_RE = re.compile(r'^UserspaceTimestamp=(?P<userspace_timestamp>.*)$', re.M)
BOOT_DELAY = datetime.timedelta(minutes=6)

class Systemd(Agent):
    def __init__(self, name, **params):
        Agent.__init__(self, name, **params)
        self.status = status.Unknown()
        self.table = TableData([
            ('status', 'état', {'align':'center'}, status_to_img),
            ('msg', 'message', {'align':'left'}, None),
            ('container', 'container', {'align':'left'}, None)
        ])
        self.data = [self.table]

    def ret_display(self, result):
        return self.display

    def measure(self):
        self.status = status.OK()
        self.display = []
        deferred = []
        cmd = getProcessOutputAndValue('/bin/systemctl', ['show', '--property=SystemState,UserspaceTimestamp'])
        cmd.addCallback(self.callback_systemd, container='root')
        deferred.append(cmd)
        if MODE_CONTENEUR_ACTIF:
            for container in GROUPS:
                if container != 'root':
                    cmd = getProcessOutputAndValue('/usr/bin/CreoleRun', ['/bin/systemctl is-system-running', container])
                    cmd.addCallback(self.callback_systemd, container=container)
                    deferred.append(cmd)
        defer = DeferredList(deferred, consumeErrors=True)
        return defer.addCallback(self.ret_display)

    def callback_systemd(self, ret, container=None):
        response, error, code = ret
        system_state = SYSTEM_STATE_RE.search(response).group('system_state')
        userspace_timestamp = USERSPACE_TIMESTAMP_RE.search(response).group('userspace_timestamp')
        if system_state == 'running':
            msg = 'démarré'
            status_ = 'On'
        else:
            status_ = 'Off'
            if system_state in ['initializing', 'starting']:
                try:
                    since = parse_time_str(userspace_timestamp)
                    if (datetime.datetime.now(tzlocal()) - since) > BOOT_DELAY:
                        self.status = status.Error()
                        msg = "Serveur toujours en cours de démarrage après {}".format(BOOT_DELAY)
                    else:
                        self.status = status.Unknown()
                        msg = "Serveur en cours de démarrage"
                except Exception as err:
                    self.status = status.Unknown()
                    msg = repr(err)
            elif system_state == 'stopping':
                self.status = status.Error()
                msg = "Serveur en cours d'arrêt"
            elif system_state == 'degraded':
                self.status = status.Error()
                msg = "Serveur opérationnel mais des services ne sont pas démarrés"
            else:
                self.status = status.Error()
                msg = "État du serveur inconnu ({})".format(response)
        self.display.append({'msg': msg, 'status': status_, 'container': container})

    def check_status(self):
        return self.status

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value
