# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Ressources implémentant les pages du site web de consultation des
agents.

Hiérarchie du site :

 - C{./} : liste des clients (C{ZephirServerResource})

 - C{./I{client}/} : liste des agents (C{ZephirClientResource})

 - C{./I{client}/I{agent}/ : détails de l'agent (C{AgentResource})

 - C{./I{client}/I{agent}/... : n'importe quel fichier dans le
   C{public_data_dir} de l'agent
"""

try: _ # localized string fetch function
except NameError: _ = str

import os
import socket
from twisted.web.resource import Resource, NoResource
from twisted.web import static
from twisted.web.util import Redirect

from zephir.monitor.agentmanager import config as cfg
from zephir.monitor.agentmanager.data import *
from zephir.monitor.agentmanager import agentdatamanager
from zephir.monitor.agentmanager.templates.AgentPageTmpl import AgentPageTmpl
from zephir.monitor.agentmanager.templates.ManagerPageTmpl import ManagerPageTmpl
from zephir.monitor.agentmanager.templates.ServerPageTmpl import ServerPageTmpl
from zephir.monitor.agentmanager.templates.FileDataTmpl import FileDataTmpl
from zephir.monitor.agentmanager.templates.RRDFileDataTmpl import RRDFileDataTmpl
from zephir.monitor.agentmanager.templates.ImgFileDataTmpl import ImgFileDataTmpl
from zephir.monitor.agentmanager.templates.HTMLDataTmpl import HTMLDataTmpl
from zephir.monitor.agentmanager.templates.MeasuresDataTmpl import MeasuresDataTmpl
from zephir.monitor.agentmanager.templates.TableDataTmpl import TableDataTmpl

module = " ({0}-{1})".format(cfg.EOLE_MODULE, cfg.EOLE_RELEASE)

class ZephirServerResource(Resource):
    """Liste de clients attachés à un serveur Zephir.

    Présente la liste des répertoires de données remontés par les
    clients.
    """

    def __init__(self, config, manager):
        Resource.__init__(self)
        self.config = config
        self.manager = manager

    def getChild(self, client_name, request):
        if client_name == "":
            return self
        elif self.manager.has_key(client_name):
            return ZephirClientResource(self.config, self.manager[client_name], client_name)
        else:
            return Redirect('/sites/%s/site/' % client_name)

    def render_GET(self, request):
        t = ServerPageTmpl()
        t.static_base = self.config['static_base_url']
        t.clients = self.manager.keys()
        return str(t)


class ZephirClientResource(Resource):
    """Liste des agents d'un client Zephir."""

    def __init__(self, config, manager, client_name):
        Resource.__init__(self)
        self.config = config
        self.manager = manager
        self.client_name = client_name
        self.show_clients = self.manager.__class__ is not agentdatamanager.LiveAgentsManager

    def getChild(self, agent_name, request):
        if agent_name == "":
            return self
        elif self.manager.has_key(agent_name):
            return AgentResource(self.config, self.manager, agent_name, self.client_name)
        else:
            return NoResource(_("No such agent"))

    def render_GET(self, request):
        t = ManagerPageTmpl()
        t.static_base = self.config['static_base_url']
        t.host = self.client_name
        t.show_clients = self.show_clients
        if module.startswith(" (zephir") and str(self.client_name) != '0':
            # affichage du numéro de serveur dans zephir
            t.hostname = "Zephir - serveur %s" % self.client_name
            t.description = t.hostname
        else:
            # récupération hostname
            # dans le cas d'une machine cible (!= zephir), on affiche le hostname et le module
            t.hostname = socket.gethostname().capitalize()
            t.description = t.hostname + module
        #t.host = self.config['host_ref']
        t.manager = self.manager # name => agent pour un client donné
        return str(t)


class AgentResource(Resource):
    """Présentation des données mesurées par un agent."""

    def __init__(self, config, agent_manager, agent_name, client_name):
        Resource.__init__(self)
        self.config = config
        self.client_name = client_name
        self.agent_manager = agent_manager
        self.agent_name = agent_name
        self.show_clients = self.agent_manager.__class__ is not agentdatamanager.LiveAgentsManager

    def getChild(self, name, request):
        agent = self.agent_manager[self.agent_name]
        path = os.path.join(cfg.agent_data_dir(self.config, self.agent_manager.client_name, self.agent_name),
                            name)
        if name == "":
            return self
        else:
            return static.File(path) # could make only one for the whole state_dir
        #         """ignore la sous-arborescence"""
        #         return self

    def render_GET(self, request):
        agent = self.agent_manager[self.agent_name]
        agent.ensure_data_uptodate()
        t = AgentPageTmpl()
        t.static_base = self.config['static_base_url']
        t.show_clients = self.show_clients
        t.host = self.client_name
        # t.host = self.config['host_ref']
        t.agent = agent
        if agent.description not in (None, ''):
            descr = agent.description
        else:
            descr = agent.name
        if module.startswith(" (zephir") and str(self.client_name) != '0':
            # affichage du numéro de serveur dans zephir
            t.hostname = "%s - Zephir - serveur %s" % (descr, self.client_name)
            t.description = t.hostname
        else:
            # récupération hostname
            # dans le cas d'une machine cible (!= zephir), on affiche le hostname et le module
            t.hostname = socket.gethostname().capitalize()
            t.description = descr + ' - ' + t.hostname + module
        t.template_for_data = lambda d: DATA_TO_TEMPLATE_MAPPING[d.__class__]()
        t.template_mapping = DATA_TO_TEMPLATE_MAPPING
        return str(t) # deferred stuff because of data generation ?



DATA_TO_TEMPLATE_MAPPING = {
    FileData: FileDataTmpl,
    RRDFileData: RRDFileDataTmpl,
    ImgFileData: ImgFileDataTmpl,
    HTMLData: HTMLDataTmpl,
    MeasuresData: MeasuresDataTmpl,
    TableData: TableDataTmpl,
    }

def template_for_data(data):
    """Renvoie le template Cheetah correspondant à un élément de
    données."""
    return DATA_TO_TEMPLATE_MAPPING[data.__class__]()



# def test_main():
#     test_support.run_unittest(UserStringTest)

# if __name__ == "__main__":
#     test_main()
