#!/bin/env python
# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# entpool.py
#
# Classes utilitaires pour la gestion d'un pool d'identifiants ENT
# sur un serveur client de Zephir
#
###########################################################################

import os, re

class Error(Exception):
    """Erreur interne
    """

class OutOfRangeError(Error):
    """valeur hors de la plage
    """

class IdPool(object):
    """classe représentant un ensemble d'identifiants spécifiques à un ENT
    les ids ont la forme suivante: LxxCiiii
    L/C : code ENT défini dans le SDET (cf annexes) et dépendant de l'entité porteuse du projet
    xx/iiii : 2 lettres et 4 chiffres identifiant un utilisateur
    """

    digits = (26, 26, 10, 10, 10, 10)

    def __init__(self, code_ent):
        max_id = 1
        for digit in self.digits:
            max_id = max_id * digit
        self.max_id = max_id - 1
        self.code_ent = code_ent
        self.digit_vals = self._def_digit_vals()
        self.regexp = re.compile('^\w[a-z]{2,2}\w\d{4,4}')

    def _def_digit_vals(self):
        """définit le poids de chaque caractère dans un identifiant
        """
        digit_vals = []
        for num_digit in range(len(self.digits)):
            val = 1
            for digit in self.digits[num_digit+1:]:
                val = val * digit
            digit_vals.append(val)
        return digit_vals

    def id_to_string(self, num_id):
        """renvoie la représentation au format SDET d'un identifiant numérique
        """
        sdet_id = []
        sdet_id.append(self.code_ent[0])
        if type(num_id) != int:
            raise TypeError("type integer requis : %s" % str(num_id))
        if 0 > num_id or num_id > self.max_id:
            raise OutOfRangeError("Valeur incompatible : %s" % str(num_id))
        for car in range(2):
            sdet_id.append(chr(ord('a') + (num_id / self.digit_vals[car])))
            num_id = num_id.__mod__(self.digit_vals[car])
        sdet_id.append(self.code_ent[1])
        for car in range(2, 5):
            sdet_id.append(str(num_id / self.digit_vals[car]))
            num_id = num_id.__mod__(self.digit_vals[car])
        sdet_id.append(str(num_id))
        return "".join(sdet_id)

    def string_to_id(self, string_id):
        """renvoie la représentation numérique d'un identifiant au format SDET
        """
        if type(string_id) not in (str, unicode):
            raise TypeError("Chaine requise : %s" % str(string_id))
        string_id = string_id.lower()
        if not self.regexp.match(string_id):
            raise ValueError("Chaine au mauvais format : %s" % string_id)
        num_id = 0
        for car in range(2):
            num_id += (ord(string_id[car + 1]) - ord('a')) * self.digit_vals[car]
        for car in range(2, 5):
            num_id += int(string_id[car + 2]) * self.digit_vals[car]
        num_id += int(string_id[7])
        return num_id

class ClientIdPool(IdPool):
    """Adaptation de la classe IdPool pour le côté client
    (maintient la liste des identifiants disponibles et
    permet leur affectation séquencielle)
    """

    def __init__(self, code_ent, state_dir="/etc/eole"):
        """initialise un pool sans identifiant disponibles
        """
        super(ClientIdPool, self).__init__(code_ent)
        self.state_file = os.path.join(state_dir, "ent_ids_%s" % code_ent)
        self.load_state()

    def load_state(self):
        """charge la liste des plages disponibles
        """
        self.free = []
        self.free_space = 0
        if os.path.isfile(self.state_file):
            data = open(self.state_file).read().strip()
            # lecture des plages correspondant au code ent
            for line in data.split('\n'):
                if line != '':
                    minid, maxid = line.split(',')
                    self._add_free_range(int(minid), int(maxid))

    def save_state(self):
        """sauvegarde les plages disponibles
        """
        try:
            f_state = open(self.state_file, 'w')
        except IOError:
            return False
        for minid, maxid in self.free:
            f_state.write('%s,%s\n' % (minid, maxid))
        f_state.close()
        return True

    def __repr__(self):
        """représentation par défaut de l'objet
        """
        descr = "ENT %s" % self.code_ent
        descr += " - %d identifiant(s) disponible(s)" % self.free_space
        if self.free:
            descr += " - prochain : %s" % self.id_to_string(self.free[0][0])
        return descr

    def add_free_range(self, minid, maxid):
        """ajoute une plage d'identifiants disponibles
        minid, maxid : identifiants au format ent
        """
        # recherche de l'emplacement où insérer la plage
        minid = self.string_to_id(minid)
        maxid = self.string_to_id(maxid)
        self._add_free_range(minid, maxid)

    def _add_free_range(self, minid, maxid):
        """ajoute une plage d'identifiants disponibles
        """
        insert_index = 0
        # mise à jour des plages libres
        if self.free:
            for rng in self.free:
                if rng[0] > maxid:
                    # on est sur la plage précédent celle à insérer
                    break
                insert_index += 1
        # ajout de la plage réservée avant insert_index
        self.free.insert(insert_index, (minid, maxid))
        self.free_space += maxid - minid + 1

    def get_next_id(self):
        """renvoie le prochain identifiant disponible
        """
        if self.free:
            first_range = self.free[0]
            new_id = first_range[0]
            # mise à jour de la plage utilisée
            if first_range[0] != first_range[1]:
                self.free[0] = (first_range[0] + 1, first_range[1])
            else:
                self.free.remove(first_range)
            self.free_space -= 1
            return self.id_to_string(new_id)
        return None
