#!/usr/bin/env python
# -*- coding: utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# manage_pool.py
#
###########################################################################

"""
librairie de fonctions pour la gestion des identifiants ENT
"""

import sys, os, base64, xmlrpclib, time, subprocess
# imports creole
from pyeole.ansiprint import print_red, print_green, print_orange
from creole.client import CreoleClient
# imports zephir
from entpool import ClientIdPool
try:
    from zephir_conf.zephir_conf import id_serveur
    from lib_zephir import zephir_proxy, convert
    registered = True
except:
    registered = False

ERR_FILE = '/etc/eole/.ent_ids_err'
OK_FILE = '/etc/eole/.ent_ids_ok'
# réserve d'identifiants à conserver dans la mesure du possible
MIN_STOCK = 50

def get_pool(code_ent=None):
    """initialisation d'un pool d'identifiant
    """
    # Récupération du code ent
    try:
        if not code_ent:
            client = CreoleClient()
            code_ent = client.get_creole('code_ent', None)
        assert len(code_ent) == 2
    except:
        return None
    # Chargement du pool d'identifiants
    return ClientIdPool(code_ent)

def get_new_ids(num_ids):
    # lecture de la cle ssh publique
    cle_pub = open('/var/spool/uucp/.ssh/id_rsa.pub').read().strip()
    try:
        code, res = convert(zephir_proxy.entid.get_id_range(id_serveur, base64.encodestring(cle_pub), num_ids))
    except xmlrpclib.ProtocolError:
        print_red("""Erreur d'authentification ou zephir non disponible""")
        return False
    if code == 0:
        print_red("Erreur lors de la réservation d'une plage d'identifiants sur Zephir: %s" % str(res))
        return False
    # confirmation de la réservation auprès de Zephir:
    # - synchronisation du serveur
    # (lancement de la lecture du fichier de plages envoyé et confirmation)
    # - attente de la fin de l'action programmée par zephir
    # - vérification des identifiants disponibles
    print_orange("Réservation d'une plage d'identifiants auprès de Zephir")
    print_orange("Veuillez patienter ...")
    orig_stdout = sys.stdout
    nullout = open('/dev/null','w')
    sys.stdout = nullout
    zephir_call = subprocess.Popen('/usr/share/zephir/scripts/zephir_client.py call >/dev/null 2>&1', shell=True)
    if zephir_call.wait() != 0:
        print_red("Erreur lors de la synchronisation avec Zéphir")
        return False
    sys.stdout = orig_stdout
    nullout.close()
    # boucle d'attente (320 secondes maxi)
    # la validation est lancée par zephir à travers uucp
    wait = 0
    while wait < 640:
        time.sleep(0.5)
        wait += 1
        if os.path.exists(OK_FILE):
            os.unlink(OK_FILE)
            return True
        if os.path.exists(ERR_FILE):
            os.unlink(ERR_FILE)
            break
    print_red("""La plage Réservée n'a pas pu être validée""")
    return False

def get_ids(num_ids, code_ent = None):
    pool = get_pool(code_ent)
    if pool is None:
        return pool
    # essai de récupération de num_ids identifiants (on essaye de conserver un stock supplémentaires)
    required = num_ids + MIN_STOCK
    if pool.free_space < required:
        if not registered:
            if pool.free_space < num_ids:
                print_red("""Pas assez d'identifiants disponibles.
Serveur non enregistré sur Zephir.
La récupération automatique d'identifiants n'est pas gérée""")
                return None
            else:
                print_orange("""Il restera moins de %s identifiants en fin de procédure""" % MIN_STOCK)
                return pool
        if pool.free_space < num_ids:
            print_orange("""Pas assez d'identifiants disponibles""")
        # calcul du nombre d'identifiants à demander (identifiants manquants + stock minimum)
        if get_new_ids(required - pool.free_space):
            print_green("Nouveaux identifiants réservés")
        # chargement du pool et vérification des identifiants disponibles après demande
        pool.load_state()
        # on vérifie qu'on a au moins le nombre d'identifiants nécessaire
        if pool.free_space >= num_ids:
            print_green("Identifiants disponibles: %s" % str(pool.free_space))
        else:
            print_red("""Impossible de réserver un nombre d'identifiants suffisants auprès de Zephir !""")
            print_red("""Identifiants disponibles: %s (%s demandés)""" % (str(pool.free_space),str(num_ids)))
            return None
    return pool

def get_single_id(code_ent = None):
    pool = get_pool(code_ent)
    if pool is None:
        return pool
    # essai de récupération d'un identifiant
    if pool.free_space < 1:
        if not registered:
            print_red("""Pas assez d'identifiants disponibles.
Serveur non enregistré sur Zephir.
La récupération automatique d'identifiants n'est pas gérée""")
            return None
        else:
            print_orange("""Pas assez d'identifiants disponibles""")
            # calcul du nombre d'identifiants à demander (identifiants manquants + stock minimum)
            if get_new_ids(MIN_STOCK):
                print_green("Nouveaux identifiants réservés")
        # chargement du pool et vérification des identifiants disponibles après demande
        pool.load_state()
        # on vérifie qu'on a au moins le nombre d'identifiants nécessaire
        if pool.free_space >= 1:
            print_green("Identifiants disponibles: %s" % str(pool.free_space))
        else:
            print_red("""Impossible de réserver de nouveaux identifiants auprès de Zephir !""")
            return None
    return pool

if __name__ == "__main__":
    print("test de génération de 50 identifiants :\n")
    pool = get_ids(50)
    if pool:
        for ent_id in range(50):
            print ent_id, ' -> ', pool.get_next_id()
    else:
        print_red("Récupération des identifiants nécessaires impossible")

