# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour les cartes réseau
"""

import re
from os.path import exists
from .ifconfig import ifconfig
from twisted.internet.utils import getProcessValue
from pyeole.process import system_out
from zephir.monitor.agentmanager import status, config
from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager.util import status_to_img, boolean_to_onoff
from zephir.monitor.agentmanager.util import log
from zephir.monitor.agentmanager.config import NOM_MACHINE, \
                                               INSTALL_RVP, \
                                               ACTIVER_HAUTE_DISPO, \
                                               NOMBRE_INTERFACES, \
                                               HAS_TYPE_AMON

class Network(Agent):

    def __init__(self, name,
                 **params):
        Agent.__init__(self, name, **params)
        self.table = TableData([
            ('status', "état", {'align':'center'}, status_to_img),
            ('interf', "Interface", {'align':'left'}, None),
            ])
        self.data = [self.table]

    def measure(self):
        self.mesures = []
        # on regarde si on est sur Amon/Sphynx ou autre
        bin_ipsec = "/usr/sbin/ipsec"
        if exists(bin_ipsec) and INSTALL_RVP:
            if ACTIVER_HAUTE_DISPO == 'non':
            # Si Amon ou Sphynx sans haute dispo, on mesure rvp
                return self.measure_rvp(bin_ipsec)
            else:
            # Si Sphynx Haute dispo, on teste le node actif
                rscfile = "/etc/ha.d/.rsc_list"
                if exists(rscfile):
                    activenode = False
                    fh = open(rscfile, 'r')
                    rsclist = fh.read().strip().split('\n')
                    for rscdesc in rsclist:
                        rsc = rscdesc.split(' ')[1]
                        findpattern = re.compile("^resource " + rsc + " is running on: " + NOM_MACHINE + " $")
                        cmd = ['crm', 'resource', 'show', rsc]
                        ret, out, err = system_out(cmd)
                        if findpattern.match(out):
                            activenode = True
                            break
                else:
                    log.msg('Fichier "' + rscfile + '" inexistant')
                    log.msg('Relancer le script "appliquer_hautedispo" pour le créer')
                    activenode = False

                if activenode:
                # Si Sphynx haute dispo et node actif, on mesure rvp
                    return self.measure_rvp(bin_ipsec)
                else:
                # Si Sphynx haute dispo et node inactif, on mesure uniquement network
                    return self.measure_network()
        else:
            return self.measure_network()

    def measure_rvp(self, cmd_rvp):
        update = getProcessValue(cmd_rvp, args=['statusall'],
                                 env={'ModeTxt': 'yes'})
        return update.addCallbacks(self.callback_rvp, self.errback_rvp)

    def callback_rvp(self, data):
        #log.msg('agent network : traitement ipsec')
        status_ipsec = 'On'
        if not HAS_TYPE_AMON or HAS_TYPE_AMON and exists('/usr/share/eole/test-rvp'):
            if data != 0:
                status_ipsec = 'Off'
            self.mesures.append({ 'interf': 'ipsec',
                                  'status': status_ipsec })
            self.measure_data['ipsec'] = status_ipsec
        return self.measure_network()

    def errback_rvp(self, err):
        log.msg('agent network : erreur, ipsec arrêté : ')
        self.mesures.append({ 'interf': 'ipsec',
                              'status': 'Off' })
        self.measure_data['ipsec'] = 'Off'
        return self.measure_network()

    def measure_network(self):
        """
        test des interfaces réseau
        """
        ic = ifconfig()
        # affiche seulement les interfaces ok (sans réslolution DNS)
        try:
            interfaces = {}
            for index_int in range(NOMBRE_INTERFACES):
                try:
                    nom_zone = getattr(config, 'NOM_ZONE_ETH{}'.format(index_int))
                    interfaces[nom_zone] = ic.getStatus(nom_zone)
                except:
                    import traceback
                    traceback.print_exc()
                    interfaces[nom_zone] = False
            interfaces_keys = list(interfaces.keys())
            interfaces_keys.sort()
            for interf in interfaces_keys:
                self.mesures.append({'interf': interf,
                                    'status': boolean_to_onoff(interfaces[interf])})
                self.measure_data[interf] = boolean_to_onoff(interfaces[interf])
        except:
            import traceback
            traceback.print_exc()
        return {'networks': self.mesures}

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value['networks']

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None:
            for service in self.last_measure.value['networks']:
                if service['status'] != 'On':
                    return status.Error()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()
