# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
###########################################################################

"""
Agent zephir pour les services distants
"""

import socket
from os import system
from creole.eosfunc import is_empty
from twisted.internet.utils import getProcessOutput
from zephir.monitor.agentmanager.agent import Agent
from zephir.monitor.agentmanager.data import TableData
from zephir.monitor.agentmanager import status
from zephir.monitor.agentmanager.config import ADRESSE_IP_DNS, \
                                               TEST_DISTANT_DOMAINE, \
                                               ACTIVER_PROXY_CLIENT, \
                                               PROXY_CLIENT_ADRESSE, \
                                               PROXY_CLIENT_PORT
from zephir.monitor.agentmanager.util import status_to_img, boolean_to_onoff

def tcpcheck(mesures, label, address, port=80, timeout=3, fallback = None):
    """tentative de connexion sur un port donné d'un serveur distant"""
    tcp = getProcessOutput('/usr/bin/tcpcheck',
                           args = [str(timeout), address+':'+str(port)],
                           env = {'LC_ALL': 'C'})
    tcp.addCallback(process_tcp, mesures, label, fallback)
    return tcp

def process_tcp(tcp_result, mesures, label, fallback):
    """ traitement du résultat de la commande """
    if b'alive' in tcp_result:
        stat = "On"
    else:
        # si il y a une adresse de 'secours', on la teste aussi
        if fallback != None:
            return tcpcheck(mesures, 'Accès Distant', fallback)
        stat = "Off"
    mesures.append({'status':stat, 'service':label})
    return mesures

def resolve(address):
    """résolution d'adresse"""
    try:
        socket.gethostbyname(address)
    except socket.error:
        return False
    return True

def testweb(mesures, label, proxy, addresses):
    """
    test des site Web d'après le code de FonctionsEoleNg
    """
    stat = "On"
    cmd_test = "http_proxy='%s' wget -q -t2 --connect-timeout=3 --delete-after %%s" % proxy
    stat = "Off"
    for address in addresses:
        if system(cmd_test % address) == 0:
            stat = "On"
            break
    mesures.append({'status':stat, 'service':label})
    return mesures

def testdns(mesures, label, dns_serv, addresses):
    """
    test des site Web d'après le code de FonctionsEoleNg
    """
    cmd_test = "/usr/bin/host -W2 -tA %s %s >/dev/null"
    stat = "Off"
    for address in addresses:
        if system(cmd_test % (address, dns_serv)) == 0:
            stat = "On"
            break
    mesures.append({'status':stat, 'service':label})
    return mesures

class Web(Agent):

    def __init__(self, name, **params):
        Agent.__init__(self, name, **params)
        self.table = TableData([
            ('status', "État", {'align':'center'}, status_to_img),
            ('service', "Service", {'align':'left'}, None),
            ])
        # récupération des DNS distant
        self.dns = ADRESSE_IP_DNS
        if is_empty(self.dns):
            self.dns = []
        # récupération du Proxy distant
        if ACTIVER_PROXY_CLIENT:
            self.proxy = (PROXY_CLIENT_ADRESSE, PROXY_CLIENT_PORT)
        else:
            self.proxy = None
        # récupération des domaines à tester
        self.domaines = TEST_DISTANT_DOMAINE
        self.data = [self.table]

    def measure(self):
        """
        Lancement des mesures
        """
        mesures = []

        ## test des DNS distants ##
        # FIXME : test inadapté : #3454
        for dns in self.dns:
            mesures = testdns(mesures, 'DNS distant : %s' % (dns), dns, self.domaines)

        ## test de la résolution ##
        stat = resolve('eole.orion.education.fr')
        mesures.append({'status':boolean_to_onoff(stat), 'service':'Résolution'})

        ## test du proxy ##
        if self.proxy is not None:
            proxy = self.proxy[0]
            port = self.proxy[1]
            tcpcheck(mesures, 'Proxy (%s)' % proxy, proxy, port)
            http_proxy = "%s:%s" % (proxy, port)
        else:
            http_proxy = ''

        ## test accès distant ##
        return testweb(mesures, 'Accès Distant', http_proxy, self.domaines)

    def write_data(self):
        Agent.write_data(self)
        if self.last_measure is not None:
            self.table.table_data = self.last_measure.value

    def check_status(self):
        """remonte une erreur si un des services est tombé"""
        if self.last_measure is not None:
            err_dns = 0
            for service in self.last_measure.value:
                self.measure_data[service['service']] = service['status']
                # cas du premier dns : on permet l'erreur
                if service['service'].startswith('DNS'):
                    if service['status'] != 'On':
                        err_dns += 1
                        # si tous les dns sont en panne : erreur
                        if err_dns == len(self.dns):
                            return status.Error()
                elif service['status'] != 'On':
                    return status.Error()
            if err_dns > 0:
                return status.Warn()
        else:
            # pas de mesure connue
            return status.Unknown()
        return status.OK()
