#!/usr/bin/env python
# -*- coding:utf-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# search_projs.py
#
# script de recherche des projets contenant des dictionnaires dans
# la forge eole (sur branche de distribution eole 2.4)
#
###########################################################################


#   Liste des exception connues (2.5.1)
#
#   dictionnaires non retrouvés dans le dépôt GIT :
#
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-bareos-mysql/24_bareos-mysql.xml  --> le répertoire devrait être mysql/dicos
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-ajaxplorer/61_ajaxplorer.xml  --> pas de branches 2.5.X (remplacé par bareos)
#
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-ovs/90_openvswitch.xml                 paquets obsolètes
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-libvirt/97_libvirt.xml                        "
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-one-market/98_one-market.xml           paquets remplacés par les modules hapy
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-one-node/98_one-node.xml                      "
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-one-flow/99_one-flow.xml                      "
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-one-frontend/99_one-frontend.xml              "
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-one-master/99_one-master.xml                  "
#    - /home/bruno/git/zephir/parc/data/dictionnaires/2.5.1/eole/eole-one-singlenode/99_one-singlenode.xml          "
#
#    Recherche des dictionnaires non référencés dans Zéphir ...
#
#    conf-amon.git                 - sondes/dicos : 11_prelude.xml    obsolète (non packagé)
#    conf-eclair.git               - dicos : 30_eclair.xml            module absent en 2.5
#    conf-zephir.git               - dicos : 30_zephir.xml            module non géré dans Zéphir
#    creole.git                    - tests/dicos : 00_base.xml        dictionnaire de tests unitaires
#    etherdraw.git                 - dicos : 61_etherdraw.xml         pas encore publié en candidate
#    zephir-client.git             - dicos : 99_zephir-stats.xml      à ignorer sur Zéphir (pas de variables ni conteneurs)
#


import urllib, glob, os, shutil, sys, argparse
from BeautifulSoup import BeautifulSoup

# A CHANGER SI BESOIN: version de distribution et branche de packaging (pour 2.4.0, renseigner 2.4)
#
# pour eole 2.4.0, seuls les paquets recompilés depuis la sortie de la version stable sont détectés

git_branches = {'2.4.0':['dist/eole/2.4.0/master'],
                '2.4.1':['dist/eole/2.4.1/master','dist/eole/2.4.0/master'],
                '2.4.2':['dist/envole/4/master', 'dist/eole/2.4.2/master', 'dist/eole/2.4.1/master', 'dist/eole/2.4.0/master'],
                '2.5.0':['dist/envole/4/master', 'dist/eole/2.5.0/master', 'dist/eole/2.4.2/master', 'dist/eole/2.4.1/master', 'dist/eole/2.4.0/master'],
                '2.5.1':['dist/envole/4/master', 'dist/eole/2.5.1/master', 'dist/eole/2.5.0/master', 'dist/eole/2.4.2/master', 'dist/eole/2.4.1/master', 'dist/eole/2.4.0/master'],
                '2.5.2':['dist/envole/5/master', 'dist/envole/4/master', 'dist/eole/2.5/master', 'dist/eole/2.5.1/master', 'dist/eole/2.5.0/master', 'dist/eole/2.4.2/master', 'dist/eole/2.4.1/master', 'dist/eole/2.4.0/master'],
                }

# 2.5.1 par défaut
# utiliser l'option l'option -d (--distrib) pour les versions antérieures
eole_version = '2.5.2'

parser = argparse.ArgumentParser(description=u"Outil de vérification des dictionnaires Zéphir depuis gitweb")

parser.add_argument('-s', '--search', action="store_true", default=False,
                    help=u"Force la recherche dans l'ensemble des projets")

parser.add_argument('-n', '--no-dl', action="store_true", default=False,
                    help=u"pas de téléchargement, utilise cache local si présent")
dist_choices = git_branches.keys()
dist_choices.sort()
parser.add_argument('--distrib', help="version de la distribution", choices=dist_choices, default=eole_version)

# URL pour vérification de l'existence d'un fichier dans un projet (utiliser .format(projet, branche, fichier)
project_url = 'http://dev-eole.ac-dijon.fr/gitweb/?p={0};a=tree;h=refs/heads/{1};hb=refs/heads/{1}'
dict_url = 'http://dev-eole.ac-dijon.fr/gitweb/?p={0};a=tree;f={2};hb=refs/heads/{1}'
access_error = '403 Forbidden - Reading tree failed'
not_found_error = '404 - Reading tree failed'
download_error = '404 - Cannot find file'
raw_dict_url = 'http://dev-eole.ac-dijon.fr/gitweb/gitweb.cgi?p={0};a=blob_plain;f={2};hb=refs/heads/{1}'

# chemin par défaut des dictionnaires dans le projet zephir. A modifier si non lancé depuis le répertoire 'outils'
tools_dir = os.path.dirname(os.path.join(os.path.abspath(__file__)))


# Projets à exclure lors de la vérification
meddee_proj = ['conf-ecdl.git', 'conf-esbl.git', 'ecdl-outils.git', 'esbl-glpi.git', 'esbl-grr.git', 'esbl-ocs.git', 'geo-ide-base.git', 'geo-ide-distribution.git', 'supervision-psin.git', 'eole-antivir2.git']
exclude_proj = ['eole-skeletor.git', 'eole-genconfig.git', 'sandbox.git', 'sandbox2.git', 'test1.git','test2.git','test3.git','test4.git','test5.git','test6.git','test7.git','test8.git','test9.git']

# dictionnaire spécifiques aux modules sur Zéphir, à ne pas prendre en compte
# lors de l'affichage des dictionnaires non retouvés dans le dépôt GIT
zephir_ignore = ['29_zephir_container.xml', '29_zephir_redefine.xml']

def get_zephirdicts(eole_version=eole_version):
    """renvoie la liste de tous les dictionnaires connues pour une version de la distribution
    """
    all_dicts = []
    paq_dicts = {}
    zephir_pool = os.path.join(os.path.dirname(tools_dir), 'data', 'dictionnaires', eole_version)
    for dict_zeph in glob.glob(os.path.join(zephir_pool, 'eole', '*', '*.xml')):
        dict_name = os.path.basename(dict_zeph)
        dict_path = os.path.dirname(dict_zeph)
        all_dicts.append(dict_name)
        paq_dicts[dict_name] = dict_path
    all_dicts.sort()
    return all_dicts, paq_dicts

def get_dictprojs(download=True, exclude=exclude_proj, eole_version=eole_version):
    """parcourt tous les projets existants et sélectionne ceux contenant des dictionnaires
    """
    download_dir = os.path.join(tools_dir, 'download_dicts_{0}'.format(eole_version))
    if download:
        if os.path.isdir(download_dir):
            if os.path.isdir("{0}_backup".format(download_dir)):
                shutil.rmtree("{0}_backup".format(download_dir))
            shutil.move(download_dir, "{0}_backup".format(download_dir))
            print "Répertoire de download précédent sauvegardé (download_dicts_backup)\n"
        os.makedirs(download_dir)

    # lecture de la liste de projets
    f_proj = urllib.urlopen('http://dev-eole.ac-dijon.fr/gitweb/')
    html_data = BeautifulSoup(f_proj.read())
    f_proj.close()

    projects = {}
    download_urls = {}
    for link in html_data.findAll('a', attrs = {'class':'list'}):
        if link.get('href').endswith('a=summary') and link.get('title') is None:
            # récupération du nom de projet
            project = link.text
            if project not in exclude:
                print 'Recherche dans ', project
                dist_found = False
                for proj_branch in git_branches[eole_version]:
                    project_tree = BeautifulSoup(urllib.urlopen(project_url.format(project, proj_branch)).read())
                    tree_body = project_tree.find('div', attrs = {'class':'page_body'})
                    if access_error not in tree_body.text and \
                       not_found_error not in tree_body.text:
                        # on a trouvé le lien vers une des branches possibles
                        dist_found = True
                        break
                if dist_found:
                    # cette branche du projet existe bien
                    dict_dirs = []
                    dicts = {}
                    proj_urls = []
                    tree = tree_body.find('table', attrs={'class':'tree'})
                    if tree:
                        # recherche des répertoire de dictionnaires ('dicos' ou '<sousprojet>/dicos')
                        for proj_file in tree.findAll('tr'):
                            name = proj_file.find('td', attrs={'class':'list'}).a.text
                            file_link = proj_file.find('td', attrs={'class':'link'}).a
                            if name and file_link and file_link.text == 'tree':
                                if not name.endswith('dicos'):
                                    name = '{0}/dicos'.format(name)
                                dict_dirs.append(name)
                        # recherche des noms de dictionnaires dans chaque répertoire détecté
                        for dict_dir in dict_dirs:
                            url_dicts = dict_url.format(project, proj_branch, dict_dir)
                            dicts_data = BeautifulSoup(urllib.urlopen(url_dicts).read())
                            dicos = []
                            for dict_link in dicts_data.findAll('a', attrs={'class':'list'}):
                                if dict_link.text.endswith('.xml'):
                                    dicos.append(dict_link.text)
                                    proj_urls.append((dict_dir, dict_link.text, proj_branch))
                                    print "- ", os.path.join(dict_dir, dict_link.text)
                            if dicos:
                                dicts[dict_dir] = dicos
                        if dicts:
                            projects[project] = dicts
                        if proj_urls:
                            download_urls[project] = proj_urls
    return projects, download_urls

def download_dicts(dl_urls, exclude=exclude_proj, eole_version=eole_version):
    download_dir = os.path.join(tools_dir, 'download_dicts_{0}'.format(eole_version))
    for project, dl_info in dl_urls.items():
        if project not in exclude:
            for dict_dir, dict_name, proj_branch in dl_info:
                print "Téléchargement de {1} ({0}) dans {2} ...".format(os.path.splitext(project)[0], dict_name, proj_branch)
                # téléchargement du fichier raw dans download_dir
                proj_dir = os.path.join(download_dir, project, dict_dir)
                if not os.path.isdir(proj_dir):
                    os.makedirs(proj_dir)
                f_dl = os.path.join(proj_dir, dict_name)
                # on vérifie que le fichier est bien trouvé
                dict_url = raw_dict_url.format(project, proj_branch, os.path.join(dict_dir, dict_name))
                dict_tree = BeautifulSoup(urllib.urlopen(dict_url).read())
                if download_error in dict_tree.contents:
                    print "Erreur 404 :", dict_url
                else:
                    # téléchargement
                    urllib.urlretrieve(dict_url, f_dl)

def check_missing(dicos, exclude=exclude_proj, only_vars=True, eole_version=eole_version):
    download_dir = os.path.join(tools_dir, 'download_dicts_{0}'.format(eole_version))
    known_dicts = set(get_zephirdicts(eole_version)[0])
    proj_names = dicos.keys()
    proj_names.sort()
    for git_proj in proj_names:
        if git_proj not in exclude:
            for dict_dir in dicos[git_proj]:
                unknown_dicts = set(dicos[git_proj][dict_dir]).difference(known_dicts)
                if unknown_dicts:
                    if only_vars:
                        unknown = []
                        for dico in unknown_dicts:
                            #content = file(os.path.join(download_dir, git_proj, dict_dir, dico)).read()
                            #if "<variable " in content or "<container " in content:
                            unknown.append(dico)
                    else:
                        unknwon = unknown_dicts
                    if unknown:
                        proj_sep = 30 - len(git_proj)
                        print "{0}- {1} : {2}".format(git_proj+" "*proj_sep, dict_dir, ', '.join(unknown))

def update_dicos(dicos, exclude=exclude_proj, eole_version=eole_version):
    # tri des dicos trouvés sur le dépôt : nom dictionnaire -> chemin dans download_dir
    # si des doublons sont trouvés, on remonte une erreur
    download_dir = os.path.join(tools_dir, 'download_dicts_{0}'.format(eole_version))
    dict_git = {}
    errors = []
    for proj, dict_infos in dicos.items():
        if proj not in exclude:
            for dict_dir, xmls in dict_infos.items():
                for dico in xmls:
                    if dico in dict_git:
                        errors.append("* le dictionnaire {0} ({1}, {2}) est déjà référencé : {3}".format(dico, proj, dict_dir, dict_git[dico]))
                    dict_git[dico] = os.path.join(download_dir, proj, dict_dir, dico)
    if errors:
        print "!! ERREURS DETECTEES !!"
        print "\n".join(errors)
        return False

    # parcours des dictionnaires déjà en place dans Zéphir
    known_dicts, paths = get_zephirdicts(eole_version)
    not_found = []
    outdated = []
    for zeph_dict in known_dicts:
        if zeph_dict in dict_git:
            # copie du dictionnaire dans le dépôt local
            dst_dict = os.path.join(paths[zeph_dict], zeph_dict)
            print "mise en place de la version du dépôt: {0} ({1})".format(zeph_dict, paths[zeph_dict].split('data/dictionnaires/')[-1])
            shutil.copy(dict_git[zeph_dict], dst_dict)
        elif zeph_dict not in zephir_ignore:
            # dictionnaire présent sur Zéphir mais non retrouvé dans les dépots
            not_found.append(os.path.join(paths[zeph_dict], zeph_dict))
    if not_found:
        # affichage des dictionnaires présents sur Zéphir mais non retrouvés
        print "\ndictionnaires non retrouvés dans le dépôt GIT :\n\n- {0}".format("\n- ".join(not_found))

if __name__ == '__main__':

    options = parser.parse_args()

    if options.distrib == "2.4.0":
        options.distrib = "2.4"

    eole_version = options.distrib

    print "DISTRIBUTION : ", eole_version
    print "BRANCHES DE PACKAGING : \n - {0}".format("\n - ".join(git_branches[eole_version]))
    print ""

    # Exclusions par défaut pour la distribution Eole
    exclusions = exclude_proj + meddee_proj

    cache_file = 'dicos_projects_{0}.py'.format(eole_version)
    if options.search or not os.path.isfile(cache_file):
        print "\nRecherche des projets contenant des dictionnaires ...\n"
        if os.path.isfile(cache_file):
            print "\nsauvegarde des données précédentes sour dicos_projects_{0}_backup.py".format(eole_version)
            shutil.copy(cache_file, 'dicos_projects_{0}_backup.py'.format(eole_version))
        dicos, dl_infos = get_dictprojs(exclude=exclusions, eole_version=eole_version)
        f_res = open(cache_file, 'w')
        f_res.write('dicos={0}'.format(dicos))
        f_res.write('\ndl_infos={0}'.format(dl_infos))
        f_res.close()
    else:
        execfile(cache_file)
        try:
            assert type(dicos) == dict
            assert type(dl_infos) == dict
        except:
            print "\nErreur : le contenu du fichier {0} est invalide"
            print "         Utiliser l'option --search pour le recréer\n".format(cache_file)
            sys.exit(1)
    print "\nliste des projets contenant des dictionnaires sauvée sous dicos_projects_{0}.py\n".format(eole_version)
    if options.no_dl == False:
        print "\nTéléchargement des dictionnaires ...\n"
        download_dicts(dl_infos, exclude=exclusions, eole_version=eole_version)
    print "\nMise à jour des dictionnaires existants ...\n"
    update_dicos(dicos, exclude=exclusions, eole_version=eole_version)
    print "\nRecherche des dictionnaires non référencés dans Zéphir ...\n"
    check_missing(dicos, exclude=exclusions, eole_version=eole_version)
    print "\nTerminé. Vérifier l'état dans le répertoire data/dictionnaires\n"

