# -*- coding: UTF-8 -*-
"""
Aggrégateur de fichiers créole
"""

import traceback
import ConfigParser
from copy import copy
from os import listdir, makedirs, unlink, minor, major, mknod, makedev, stat
from stat import S_IFBLK
from os.path import isfile, isdir, join, dirname, ismount, exists
from glob import glob
from commands import getstatusoutput
from shutil import move, copy as shutil_copy
from tempfile import mktemp
from time import sleep
try:
    import json
except:
    import simplejson as json

from creole2.log import logger
from creole2.error import FileNotFound, ConfigError, DependencyError, \
TypeEoleError
from creole2.lxml_parser import parse_xml_file, parse_string
from creole2.template import Template
from creole2.typeole import type_factory
from creole2 import dico, config
# Importation des fonctions accessibles depuis les templates (eval)
from creole2 import eosfunc
from creole2.eosfunc import load_container_var
from pyeole2.service import service_code, service_out, update_rcd
from pyeole2.process import system_code, system_out, is_installed, tcpcheck
from creole2.utils import encode_list, classes, stringify
from pyeole2.ansiprint import print_cyan, print_orange

class EoleDict:
    """
    Aggrégateur de fichier eol
    """

    def __init__(self, templatedir=config.templatedir, logfile=None, linter=False):
        """
        files: nom long des fichiers cibles
        variables: dictionnaire des variables
        Les variables sont empilées dans des listes.

        @param templatedir: répertoire des fichiers template
        @param log: instance de logger
        """
        #flag specifique pour creolelint
        self.linter = linter
        self.store = ConfigParser.ConfigParser(dict_type=dict)
        self.files = {}
        self.files_index = []
        self.containers = {}
        self.filelists = {}
        self.variables = {}
        self.families = {} # all families
        self.normal_families = [] # only non expert mode families
        self.family_order = []
        self.constraints = {}
        self.depends = {} # variables depending on other variables
        self.orphan_constraints = {'checks':{}, 'fills':{}, 'auto':{}}
        self.auto_vars = {}
        self.groups = {}
        self.helps = {'variables':{}, 'families':{}}
        self.separators = {}
        self.templatedir = templatedir
        self.log = logger
        self.virt_not_load = []
        if logfile == None:
            self.log.set_logfile()
        else:
            self.log.set_logfile(logfile)

    def read(self, filename):
        """Aggregation d'un fichier dictionnaire
        @param filename: fichier de configuration
        """
        self.log.debug('#-> entering read()' + filename)
        try:
            # on vérifie que le fichier existe
            if not isfile(filename):
                raise FileNotFound, "le fichier %s n'existe pas" % filename

            #parsing du fichier
            new_softwares, new_variables, new_families, \
                new_order, new_constraints, new_groups, helps, \
                separators = parse_xml_file(filename)
            self.log.info('lecture du fichier  %s ' % filename)

        except Exception, err:
            self.log.error(traceback.format_exc())
            raise ConfigError, "erreur au chargement d'une configuration : %s" % str(err)

        return self._read(new_softwares, new_variables, new_families, new_order,
                          new_constraints, new_groups, helps, separators)

    def import_values(self, filename):
        """
        import de valeurs depuis un fichier creole 2.2
        """
        from creole2.upgrade import upgrade
        if not isfile(filename):
            raise FileNotFound, "le fichier %s n'existe pas" % filename
        # le traitement se fait directement avec Cfgparser
        self.store._sections = upgrade(filename)
        self._load_values(check=False)

    def import_values_creole1(self, filename):
        """import de valeurs depuis un fichier creole1
        """
        errors = []
        warnings = []
        try:
            if not isfile(filename):
                raise FileNotFound, "le fichier %s n'existe pas" % filename
        except Exception, err:
            raise ConfigError, "erreur au chargement d'une configuration : %s" % str(err)
        # chargement du dictionnaire
        old_dict = dico.DicoEole(filename)
        # essai de détection du module Eole d'origine
        module = dico.detect_module(old_dict.liste_vars.keys())
        if module ==  '':
            raise ConfigError, "erreur de détection du type de module"
        # chargement du dictionnaire des conversion des variables 1 -> 2
        dict_convert = dico.read_convert(join(config.datadir, 'convert',
            '%s.csv' % module))
        # import des valeurs
        for newvar in dict_convert.keys():
            val = []
            # la valeur doit être récupérée dans l'ancienne conf
            for old_var in dict_convert[newvar]:
                try:
                    old_val = old_dict.get_value(old_var)
                except:
                    # variable non renseignée dans l'ancien dictionnaire
                    old_val = ''
                if old_val != '':
                    val.append(old_val)
                # insertion de la valeur
                try:
                    self.set_value(newvar, val)
                except KeyError, err:
                    errors.append('variable inconnue : %s' % str(err))
                except Exception, err:
                    errors.append(str(err))

        # on recalcule les valeurs pour toutes les autres variables
        for family in self.family_order:
            for newvar in self.families[family]['vars']:
                if newvar not in dict_convert.keys():
                    try:
                        val = self.get_value(newvar)
                        self.set_value(newvar, val)
                    except TypeEoleError, err:
                        warnings.append(str(err))
            return errors, warnings

    def read_string(self, xml_string):
        """Aggregation d'un fichier dictionnaire
        @param xml_string: fichier de configuration
        """
        try:
            if not xml_string.startswith("<?xml"):
                raise FileNotFound, "le dictionnaire doit être au format xml"
            #parsing de la chaîne
            #FIXME !!!
            new_files, new_variables, new_families, new_order, \
                new_constraints, new_groups, helps, \
                separators = parse_string(xml_string)
        except Exception, err:
            raise ConfigError, "erreur au chargement d'une configuration : %s" % str(err)
        return self._read(new_files, new_variables, new_families, new_order,
                          new_constraints, new_groups, helps, separators)


    def _read_softwares(self, softwares):
        """
        traitement des balises files, packages et services
        """
        cvar = eosfunc.load_container_var()
        #software = [files, packages, services]
        for name, software in softwares.items():

            ## traitement des fichiers ##
            for fic, value in software[0].items():
                if config.VIRTDISABLED == False and name != config.VIRTMASTER:
                    #fic = join(config.VIRTROOT, name, config.VIRTBASE, fic.lstrip('/'))
                    try:
                        fic = join(cvar['container_path_%s'%name], fic.lstrip('/'))
                    except:
                        fic = join(config.VIRTROOT, name, config.VIRTBASE, fic.lstrip('/'))
                if config.VIRTDISABLED == True and value.get('container_only', False) == True:
                    continue
                template = Template(fic, value['source'],
                                    templatedir = self.templatedir,
                                    mode=value['mode'],
                                    owner=value['owner'],
                                    group=value['group'],
                                    mkdir=value['mkdir'],
                                    container = name,
                                    log=self.log.filename,
                                    del_comment=value['del_comment'])
                self.files[fic] = {'var':template, 'hidden':value['hidden'],
                        'rm':value['rm'], 'container': name}
                self.files_index.append((value['level'], fic))
                if value['filelist'] != '':
                    self.filelists.setdefault(value['filelist'], []).append(fic)
            if not name in self.containers:
                self.containers[name] = {'packages':[], 'services':{},
                                         'interfaces': {}, 'id': '',
                                         'disknods':[], 'name':name}
            if software[1] != []:
                self.containers[name]['packages'] += software[1]

            ## traitement des services ##
            for servicename, value in software[2].items():
                servicelist = value['servicelist']
                startlevel = value['startlevel']
                stoplevel = value['stoplevel']
                level = value['level']
                method = value['method']
                pty = value['pty']
                active = not value['hidden']
                in_container = value['in_container']
                if in_container is not None:
                    if in_container and config.VIRTDISABLED:
                        # in_container='True' et mode non conteneur
                        continue
                    if not in_container and not config.VIRTDISABLED:
                        # in_container='False' et mode conteneur
                        continue
                if self.containers[name]['services'].has_key(servicename):
                    if not value['redefine']:
                        raise Exception('Le service {0} existe déjà dans le conteneur {1}'.format(servicename, name))
                    else:
                        self.containers[name]['services'].pop(servicename)
                elif value['redefine']:
                    raise Exception("Le service {0} n'existe pas dans le conteneur {1} alors qu'il est redefini".format(servicename, name))
                self.containers[name]['services'][servicename] = {'active': active,
                        'servicelist': servicelist,
                        'startlevel': startlevel,
                        'stoplevel': stoplevel,
                        'method': method,
                        'pty': pty,
                        'level': level}

            ## traitement des interfaces ##
            if not self.containers[name].has_key('interfaces'):
                self.containers[name]['interfaces'] = {}
            for interfacename, value in software[3].items():
                self.containers[name]['interfaces'][interfacename] = {'active': True,
                        'interfacelist': value['interfacelist'],
                        'linkto': value['linkto'],
                        'ip': value['ip'],
                        'mask': value['mask'],
                        'bcast': value['bcast'],
                        'method': value['method'],
                        }
            if software[4] != '':
                self.containers[name]['id'] = software[4]
            if software[5] != '':
                self.containers[name]['group'] = software[5]
            if software[6] != []:
                self.containers[name]['disknods'] = software[6]
            self.containers[name]['name'] = name
            #self.containers[name]['services'].append(software[2])

    def _valid_read_variable(self, var):
        """
        vérification de la validité des variables déclarées
        """
        for varname in var.keys():
            if self.variables.has_key(varname):
                if not var[varname]['exists']:
                    #exists == True => il faut que la variable soit definie
                    #donc si la nouvelle variable n'existe pas dans le dico EOLE
                    # elle ne doit pas etre prise en compte
                    continue
                if not var[varname]['redefine']:
                    # on de devrait pas avoir 2 fois la meme variable
                    raise ConfigError, "deux variables ont le même nom (%s)" % str(varname)
            else:
                if var[varname]['redefine']:
                    if self.linter == False:
                        raise ConfigError, "variable %s redefinie mais non existante" % str(varname)
                    else:
                        if var[varname]['type'] == '':
                            var[varname]['type'] = 'string'

    def _read(self, new_softwares, new_variables, new_families, new_order, new_constraints, new_groups, helps, separators):
        """charge les données venant d'un dictionnaire"""
        # on commence par vérifier qu'il n'y a pas de doublons (variables)
        self._valid_read_variable(new_variables)

        # ajout des nouveaux fichiers/paquets et services dans le dictionnaire final
        self._read_softwares(new_softwares)

        # ajout des nouvelles familles
        for family, data in new_families.items():
            if data['mode'] != 'expert' and family not in self.normal_families:
                # gestion de la liste des familles "normales"
                # -> normal un jour, normal toujours
                self.normal_families.append(family)
            #si famille inexistante, on la crée mais sans les variables
            if family not in self.families:
                # évidemment, il faut faire une copie !
                self.families[family] = copy(data)
                self.families[family]['vars'] = []
            for var in data['vars']:
                #si on redefinit une variable, la variable change aussi de famille si nécessaire
                if new_variables[var]['redefine']:
                    for tfamily in self.families.keys():
                        if var in self.families[tfamily]['vars'] and family != tfamily:
                            self.families[tfamily]['vars'].remove(var)
                            self.families[family]['vars'].append(var)
                            break
                #autres cas : ajout si nécessaire
                elif not var in self.families[family]['vars']:
                    self.families[family]['vars'].append(var)
                    #exists=True : annulation si nouvelle famille (#3017)
                    if not new_variables[var]['exists']:
                        for tfamily in self.families.keys():
                            if var in self.families[tfamily]['vars'] and family != tfamily:
                                self.families[family]['vars'].remove(var)
                                break
                #else: normalement c'est le cas exists='False' déjà déclaré

        # stockage de l'ordre des variables
        for label in new_order:
            if label not in self.family_order:
                self.family_order.append(label)

        # ajout des nouvelles variables dans le dictionnaire final
        for key, values in new_variables.items():
            #si la derniere variable ne devait pas etre prise en compte
            #existe == True => on quitte brutalement le for
            #et il ne faut pas prendre en compte la suite :
            if not new_variables[key]['exists'] and self.variables.has_key(key):
                continue
            if not new_variables[key]['redefine']:
                # cas normal sans redefine
                self.variables[key] = self._create_var(key, values, new_constraints)
                # hack pour gérer mandatory='True' (FIXME: dans redefine?)
                if new_variables[key]['mandatory']:
                    self.variables[key].checks.append(['obligatoire', []])
            else:
                # cas redefine='True'
                if not self.variables.has_key(key):
                    #pour creolelint
                    self.variables[key] = self._create_var(key, values, new_constraints)
                #uniquement dans le cadre de redefine
                if new_variables[key]['value'] != []:
                    self.variables[key].valeole = encode_list(new_variables[key]['value'])
                    self.variables[key].val = self.variables[key].valeole
                if new_variables[key]['description'] != '':
                    self.variables[key].description = new_variables[key]['description']
                self.variables[key].hidden = new_variables[key]['hidden']
                #la propriété multi n'est pas modifiable (#4059)
                #self.variables[key].multi = new_variables[key]['multi']
                #if new_constraints['auto'].has_key(key):
                #    self.variables[key].auto.extend(new_constraints['auto'][key])
                if new_variables[key]['remove_check']:
                    #si remove_check, supprime tous les anciens check
                    self.variables[key].checks = []
                if new_constraints['checks'].has_key(key):
                    # si on redefinie les valid_enum, supprime d'abord l'ancien valid_enum
                    new_valid = False
                    for valid in new_constraints['checks'][key]:
                        if u'valid_enum' in valid:
                            new_valid = True
                            break
                    if new_valid:
                        for valid in copy(self.variables[key].checks):
                            if u'valid_enum' in valid:
                                self.variables[key].checks.remove(valid)
                    #self.variables[key].checks.extend(new_constraints['checks'][key])
                if new_constraints['conditions'].has_key(key):
                    # si on redefini les hidden_if_in, on supprime d'abord l'ancien hidden_if_in
                    new_hidden_if_in = False
                    new_hidden_if_not_in = False
                    for valid in new_constraints['conditions'][key]:
                        if u'hidden_if_in' in valid:
                            new_hidden_if_in = True
                        if u'hidden_if_not_in' in valid:
                            new_hidden_if_not_in = True
                    if new_hidden_if_in:
                        if self.constraints.has_key(key):
                            for valid in copy(self.constraints[key]):
                                if u'hidden_if_in' in valid:
                                    self.constraints[key].remove(valid)
                    if new_hidden_if_not_in:
                        if self.constraints.has_key(key):
                            for valid in copy(self.constraints[key]):
                                if u'hidden_if_not_in' in valid:
                                    self.constraints[key].remove(valid)
                    #if not self.constraints.has_key(key):
                    #    self.constraints[key] = []
                    #self.constraints[key].extend(new_constraints['conditions'][key])
                self.variables[key].mode = new_variables[key]['mode']

        # si des contraintes sont présentes pour des variables définies
        # dans un autre dictionnaire
        # elles sont affectées une fois les variables disponibles
        # les contraintes orphelines sont stockées sous le format suivant:
        # self.orphan_constraints = {'type': [(set(vars_manquantes), cible, contrainte), (...)]}}
        for type_const in ['checks', 'fills', 'auto']:
            upd_orphans = []
            # on vérifie les contraintes orphelines précédentes
            for missing, target, constr in self.orphan_constraints.get(type_const, []):
                # on retire les nouvelles variables des variables manquantes
                missing.difference_update(set(new_variables.keys()))
                if not missing:
                    # toutes les variables sont présentes
                    self.update_constraints(target, type_const, constr)
                else:
                    upd_orphans.append((missing, target, constr))
            self.orphan_constraints[type_const] = upd_orphans
            # on recherche les nouvelles contraintes orphelines
            for var, liste_consts in new_constraints.get(type_const, {}).items():
                for const in liste_consts:
                    external_vars = []
                    # vérification des variables utilisées en paramètre (paramètre de type 'eole')
                    for param in const[1]:
                        if param['type'] == 'eole':
                            # on stocke la dépendance entre les 2 variables
                            self.add_depends(var, param['value'])
                            if param['value'] not in new_variables:
                                # variable non présente dans ce dictionnaire,
                                if not param['optional']:
                                    # on la considère manquante si elle n'est pas optionelle
                                    external_vars.append(param['value'])
                    # vérification de la variable cible
                    if var not in new_variables:
                        external_vars.append(var)
                    external_vars = set(external_vars)
                    external_vars.difference_update(set(self.variables.keys()))
                    if external_vars:
                        # variables externe non chargée: on stocke
                        orphans = self.orphan_constraints.get(type_const, [])
                        orphans.append((external_vars, var, const))
                        self.orphan_constraints[type_const] = orphans
                    else:
                        # nouvelle contrainte à affecter à une variable déjà chargée
                        self.update_constraints(var, type_const, const)
        # ajout des contraintes
        # FIXME : #754
#        if redefine == False:
        for var, conditions in new_constraints['conditions'].items():
            self.constraints.setdefault(var, []).extend(conditions)
        for var, auto_funcs in new_constraints['auto'].items():
            for func in auto_funcs:
                for param in func[1]:
                    if param['name'] == 'parametre':
                        self.auto_vars.setdefault(param['value'], []).append(var)

        # ajout des groupes
        for key, values in new_groups.items():
            if self.groups.has_key(key):
                # extension d'un groupe existant
                self.groups[key].extend(values)
                #raise ConfigError, "groupe deja existant : %s" % key
            else:
                self.groups[key] = values
            # ajout du nom du maitre dans les objets esclaves
            for val in values:
                self.variables[val].slave = key
                self.add_depends(val, key)
        # stockage de l'aide
        for key, values in helps['variables'].items():
            if self.helps['variables'].has_key(key):
                raise ConfigError, "aide déjà existante pour : %s" % str(key)
            else:
                self.helps['variables'][key] = values
        for key, values in helps['families'].items():
            if self.helps['families'].has_key(key):
                raise ConfigError, "aide déjà existante pour : %s" % str(key)
            else:
                self.helps['families'][key] = values

        # séparateurs (affichage)
        for var, value in separators.items():
            if self.separators.has_key(var) and new_variables[var]['exists']:
                raise ConfigError, "plus d'un separateur défini pour : %s" % str(var)
            else:
                self.separators[var] = value

        # on revérifie toutes les contraintes de condition après chargement
        for var in self.constraints.keys():
            try:
                self.check_conditions(var)
            except KeyError:
                # on laisse passer si la cible n'est pas présente
                # (permet de mettre une condition sur un élément d'un autre
                # dictionnaire sans se soucier de l'ordre de chargement)
                self.log.error(traceback.format_exc())
                raise Exception('Erreur : variable non trouvee')
        # On initialise les valeurs par défaut (les fonctions 'fills' orphelines
        # ne sont pas prises en compte tant que des variables manquent)
        for var in self.variables.values():
            var.init_fills()

    def read_dir(self, dir_config):
        """
        lecture d'un répertoire entier de dictionnaires
        """
        # chargement des dictionnaires
        if type(dir_config) != list:
            dir_config = [dir_config]
        for mydir in dir_config:
            if isdir(mydir):
                fichiers = listdir(mydir)
                fichiers.sort()
                for dic in fichiers:
                    if dic.endswith('.xml'):
                        try:
                            self.read(join(mydir, dic))
                        except Exception, err:
                            self.log.error(traceback.format_exc())
                            traceback.print_exc()
                            raise ConfigError, 'erreur lors du chargement de %s : %s' % (join(mydir, dic), str(err))

    def add_depends(self, var, depends):
        depends_on = self.depends.get(var, [])
        if depends not in depends_on:
            depends_on.append(depends)
        self.depends[var] = depends_on

    def _create_var(self, name, var, constraints):
        """ recherche des fonctions liées à cette variable """
        checks = []
        fills = []
        auto = []
        # les contraintes seront prises en compte au fur et à mesure du chargement
        # des dictionnaires (quand les variables nécessaires sont disponibles)
        eolevar = type_factory(
                       context = self,
                       mime = var['type'],
                       name = name,
                       valeole = var['value'],
                       description = var['description'],
                       hidden = var['hidden'],
                       multi = var['multi'],
                       checks = checks,
                       fills = fills,
                       auto = auto,
                       linter = self.linter,
                       mode = var['mode']
                       )

        return eolevar

    def update_constraints(self, var, type_const, constr):
        """ mise à jour des contraintes d'une variable """
        if hasattr(self.variables.get(var), type_const):
            getattr(self.variables[var], type_const).append(constr)
        else:
            setattr(self.variables[var], type_const, [constr])

    def set_value(self, var, values, default=False, force=False):
        """ affectation d'une valeur à une variable """
        # actualisation de la valeur
        self.variables[var].set_value(values, default, force=force)
        # vérification des dépendances
        self.check_conditions(var, force)

    def get_value(self, var, default=False):
        """ lecture de la valeur d'une variable """
        val = self.variables[var].get_value(default)
        if val == []:
            return ['']
        else:
            return val

    def get_final_value(self, var):
        return self.variables[var].get_final_value()

    def get_prec_value(self, var):
        val = self.variables[var].get_prec_value()
        if val == []:
            return ['']
        else:
            return val

    def get_description(self, var):
        return self.variables[var].description

    def instance(self, container=None, debug=False):
        """
        Instancie les fichiers selons les valeurs des variables EOLE
        """
        self.log.info('===========================================')
        self.log.info('        Debut de l\'instanciation')
        self.log.info('===========================================')

        for fic, value in self.files.items():
            if value['hidden'] == False:
                value['var'].verify()
        # parcourt la liste des fichiers dans l'ordre renseigne
        # dans les dictionnaires (#2057)
        self.files_index.sort()
        for level, fic in self.files_index:
            value = self.files[fic]
            if container != None and value['container'] != container:
                continue
            if not value['hidden']:
                value['var'].process(self)
            elif value['rm']:
                value['var'].remove_destfile()
                self.log.info('Fichier %s supprime' % fic)
            else:
                self.log.info('Fichier %s non instancie'%fic)
        self.log.info('===========================================')
        self.log.info('        Fin de l\'instanciation')
        self.log.info('===========================================')

    def get_container_name(self, name, tcontainer):
        if tcontainer[name].has_key('group') and \
                    tcontainer[name]['group'] != name:
            cname = self.get_container_name(tcontainer[name]['group'], tcontainer)
        else:
            cname = name
        return cname

    def _make_mac_address(self, interface_master, interface_slave, id_container):
        # Génération d'adresse mac à partir de l'oid de Virtual Iron (00:0F:4B) et de l'ip du maitre
        # converti en hexdecimal et de l'interface du conteneur converit en hexadecimal et de l'id du conteneur
        # converti en hexa
        from random import randint
        try:
            val_ip = self.variables['adresse_ip_%s' % interface_master].get_value()[0]
            val_ip_splitted = val_ip.split('.')
            if len(val_ip_splitted) != 4:
                raise Exception("adresse_ip_%s n'est pas une IP valide : \"%s\"" % (interface_master, val_ip))
            ip_master = val_ip_splitted[3]
            interface = interface_slave[3:]
            ip_master = "%x" % int(ip_master)
            if len(ip_master) == 1:
                ip_master = '0' + ip_master
            id_container = "%x" % int(id_container)
            if len(id_container) == 1:
                id_container = '0' + id_container
            interface = "%x" % int(interface)
            if len(interface) == 1:
                rand = '%x' % randint(0, 15)
                interface = rand + interface
            mac = '00:0F:4B:%s:%s:%s' % (ip_master.upper(), interface.upper(), id_container.upper())
        except Exception, e:
            #import traceback
            #traceback.print_exc()
            raise Exception("Impossible de calculer l'adresse mac : %s " % e)
        return mac

    def _make_root_path(self, name):
        return join(config.VIRTROOT, str(name))

    def test_container_conf(self):
        if not isfile(config.containers_file):
            raise FileNotFound, "le fichier %s n'existe pas" % \
                    config.containers_file
        containers_file_content = {}
        execfile(config.containers_file, {}, containers_file_content)
        for name in self.containers:
            path = "container_path_{0}".format(name)
            ip = "container_ip_{0}".format(name)
            #container_name_root n'existait pas, il faut gérer la migration
            if name == 'root':
                cname = ip
            else:
                cname = "container_name_{0}".format(name)
            for var in [path, ip, cname]:
                if var not in containers_file_content:
                    raise ConfigError(
                            'Variable {0} non présente dans {1}'.format(
                                var, config.containers_file))

    def forcestart_container(self):#, inputfile, outputfile):
        cmd = ['/etc/init.d/lxc', 'start']
        system_code(cmd)
        gcontainers = set()
        cvar = eosfunc.load_container_var()
        for container in self.containers:
            if container != config.VIRTMASTER:
                name = cvar["container_name_%s" % container]
                ip = cvar["container_ip_%s" % container]
                gcontainers.add((name, ip))

        all_ok = 'notok'
        timer = 0
        while all_ok != '':
            timer += 1
            sleep(1)
            all_ok = ''
            for container, ip in gcontainers:
                if not tcpcheck(ip, '22'):
                    all_ok = container
                    break
            if all_ok == '':
                break
            if timer == 14:
                print_orange("Impossible de contacter le conteneur {0}, ouverture du firewall".format(all_ok))
                system_out('/usr/sbin/ouvre.firewall')
            if timer >= 15:
                break

        if all_ok != '':
            raise Exception("Le conteneur {0} n'est pas accessible".format(
                all_ok))
        print "Les conteneurs sont correctement démarrés"

    def write_config(self):
        #vider et creer le fichier containers_bridge_file
        cb_fh = open(config.containers_bridge_file, 'w')
        cb_fh.write('# fichier container bridge\n')

        if config.VIRTDISABLED == True:
            cb_fh.close()
            return
        tcontainer = self.containers
        containers = []
        for name in tcontainer.keys():
            if name != config.VIRTMASTER:
                containers.append(self.get_container_name(name, tcontainer))
        containers = list(set(containers))
        #Regeneration du fichier hosts dans les conteneurs
        for container in containers:
            filename = join(self._make_root_path(container), 'hosts.d', 'default')
            if isfile(filename):
                unlink(filename)
            for filename in glob(join(self._make_root_path(container), 'hosts.d', 'eth*')):
                unlink(filename)
        #boucle sur les conteneurs pas sur les groupes
        for container in tcontainer:
            gname = self.get_container_name(container, tcontainer)
            for interface, values in self.containers[container]['interfaces'].items():
                if values['active']:
                    self._write_hosts_d(gname, container, '%%%%%s'%values['ip'], interface)
            lxc_hosts_file = join("/usr/share/eole/lxc/hosts", container)
            if isfile(lxc_hosts_file):
                fh = open(join(self._make_root_path(gname), 'hosts.d', 'default'), 'a')
                fh.write(open(lxc_hosts_file).read())
                fh.close()

        for container in containers:
            if not isdir(self._make_root_path(container)):
                raise Exception("Le conteneur %s n'existe pas" % container)
            for filename in glob(join(self._make_root_path(container), 'config.d', 'eth*')):
                unlink(filename)
            for filename in glob(join(self._make_root_path(container), 'interfaces.d', 'eth*')):
                unlink(filename)
            #pour migration après résolu #1806
            if not isdir(join(self._make_root_path(container), 'interfaces.d')):
                makedirs(join(self._make_root_path(container), 'interfaces.d'))
                move(join(self._make_root_path(container), 'rootfs', 'etc', 'network', 'interfaces'),
                        join(self._make_root_path(container), 'interfaces.d', 'internal'))

        containers_bridge = {}
        mac = {}
        for name, value in tcontainer.items():
            if name != config.VIRTMASTER:
                cname = self.get_container_name(name, tcontainer)
                for interface, values in self.containers[name]['interfaces'].items():
                    if values['active']:
                        if values['linkto'] in self.variables:
                            # si 'linkto' correspond à une variable, on prend sa valeur
                            linkto = self.variables[values['linkto']].get_value()[0]
                        else:
                            # si pas de variable, on utilise linkto tel quel
                            # FIXME :(attention si une variable est nommée eth0, eth1,
                            # ... le test sera cassé, utiliser une variable dans tous les cas ?)
                            linkto = values['linkto']
                        code, stdout, stderr = system_out(['ifconfig', linkto])
                        if code != 0:
                            raise Exception("problème avec l'interface {0} : {1}".format(linkto, stderr))
                        self._write_interface_d(values['ip'], values['mask'], values['bcast'], linkto, values['method'], interface, cname)
                        if not mac.has_key(cname):
                            mac[cname] = {}
                        mac[cname][interface] = self._make_mac_address(linkto, interface, tcontainer[cname]['id'])
                        if values['method'] == 'bridge':
                            if linkto not in containers_bridge.keys():
                                containers_bridge[linkto] = [self._build_ve_interface(cname, interface)]
                            else:
                                containers_bridge[linkto].append(self._build_ve_interface(cname, interface))
        #ecriture du fichier des bridges
        #suppression des doubles
        for linkto, interfaces in containers_bridge.items():
            cb_fh.write("#set global $method_{0}='bridge'\n".format(linkto))
            cb_fh.write("#set global $br_bridge_{0}={1}\n".format(linkto, interfaces))
        cb_fh.close()
        rep = []
        for container in containers:
            rep.append((join(self._make_root_path(container), 'config.d'),
                        join(self._make_root_path(container), 'config'),
                        container))
        for inputdir, outputfile, cname in rep:
            if mac.has_key(cname):
                for interface, macaddr in mac[cname].items():
                    try:
                        name = 'adresse_mac_%s' % interface
                        macadr = type_factory(
                        context = self,
                        mime = "string",
                        name = name,
                        val = [macaddr],
                        description = "adresse mac",
                        )

                        self.variables[name] = macadr
                    except:
                        traceback.print_exc()
            inputfile = mktemp()
            files = glob(join(inputdir, '*'))
            fd = open(inputfile, 'wa')
            for filename in files:
                fd.write(open(filename, 'r').read())
            fd.close()
            datadir = dirname(inputfile)
            t = Template(inputfile, templatedir=datadir)
            t.target = outputfile
            t.process(self)
            unlink(inputfile)

        for container in containers:
            inputfile = mktemp()
            files = glob(join(self._make_root_path(container), 'interfaces.d', '*'))
            fd = open(inputfile, 'wa')
            for filename in files:
                fd.write(open(filename, 'r').read())
            fd.close()
            datadir = dirname(inputfile)
            t = Template(inputfile, templatedir=datadir)
            t.target = join(self._make_root_path(container), 'rootfs',
                            'etc', 'network', 'interfaces')
            t.process(self)
            unlink(inputfile)

        for container in containers:
            inputfile = mktemp()
            inputdir = join(self._make_root_path(container), 'hosts.d')
            outputfile = join(self._make_root_path(container), config.VIRTBASE, 'etc/hosts')
            files = glob(join(inputdir, '*'))
            fd = open(inputfile, 'wa')
            for filename in files:
                fd.write(open(filename, 'r').read())
            fd.close()
            datadir = dirname(inputfile)
            t = Template(inputfile, templatedir=datadir)
            t.target = outputfile
            t.process(self)
            unlink(inputfile)
        self.write_fstab_config()

    def write_fstab_config(self):
        tcontainer = self.containers
        containers = []
        for name in tcontainer.keys():
            if name != config.VIRTMASTER:
                containers.append(self.get_container_name(name, tcontainer))
        containers = list(set(containers))

        #vide le répertoire
        for container in containers:
            destdir = join(self._make_root_path(container), 'fstab.d')
            if not isdir(destdir):
                makedirs(destdir)
            for filename in glob(join(destdir, '*')):
                unlink(filename)

        #remplit le répertoire
        for container in tcontainer:
            if container == 'root':
                continue
            cname = self.get_container_name(container, tcontainer)
            destdir = join(self._make_root_path(cname), 'fstab.d')
            #copie une fois le fichier DEFAULT par groupe de conteneur
            if not isfile(destdir):
                shutil_copy(join(config.container_fstab_dir, 'DEFAULT'),
                        destdir)
            #copie le fichier fstab dans le groupe de conteneur
            cfstab = join(config.container_fstab_dir, container)
            if isfile(cfstab):
                shutil_copy(cfstab, destdir)

        #concat les fichiers
        for container in containers:
            destfile = join(self._make_root_path(container), 'fstab')
            if isfile(destfile):
                oldfile = open(destfile, 'r').read()
            else:
                oldfile = None
            destdir = join(self._make_root_path(container), 'rootfs')
            inputfile = mktemp()
            files = glob(join(self._make_root_path(container), 'fstab.d', '*'))
            fd = open(inputfile, 'wa')
            for filename in files:
                #il faut garder {{ROOTFS}} parce que peut etre faire avant
                #gen_config
                content = open(filename, 'r').read()
                content = content.replace('{{ROOTFS}}', destdir)
                fd.write(content)
            fd.close()
            #test si déjà eu une instance
            if isfile(config.configeol):
                datadir = dirname(inputfile)
                t = Template(inputfile, templatedir=datadir)
                t.target = destfile
                t.process(self)
            else:
                #si pas d'instance, supprimer les lignes templatisées
                fd = open(destfile, 'wa')
                content = open(inputfile, 'r').read()
                for line in content:
                    if '%' not in line:
                        fd.write(line)
                fd.close()

            unlink(inputfile)
            #une fois templatisé, reouvre le fichier pour créer
            #les répertoires si nécessaire
            for line in open(destfile, 'r').readlines():
                try:
                    mountpoint = line.split()[1]
                    if not isdir(mountpoint):
                        makedirs(mountpoint)
                except:
                    import traceback
                    traceback.print_exc()
                    pass

            #test si modification dans le fichier fstab si le conteneur
            #n'est pas arrêté
            #code, stdout, stderr = system_out(['/usr/bin/lxc-info', '-n', container])
            #if stdout != "'{0}' is STOPPED\n".format(container):
            #    newfile = open(destfile, 'r').read()
            #    if oldfile and oldfile != newfile:
            #        print_orange("Nouveau montage détecté, un redémarrage du conteneur {0} est recommandé".format(container))

    def _build_ve_interface(self, cname, interface):
        #eth1 => 1
        veinterface = interface[3:]
        return 've%s%s' % (cname, veinterface)

    def _write_interface_d(self, ip, mask, bcast, linkto, method, interface, cname):
        file_path = join(self._make_root_path(cname), 'config.d', interface)
        fd = open(file_path, 'w')
        if method == 'macvlan':
            fd.write("""
lxc.network.type=macvlan
lxc.network.link= %s
lxc.network.name= %s
lxc.network.ipv4 = %%%%%s/%%%%calc_classe(%%%%%s)
lxc.network.hwaddr = %%%%adresse_mac_%s
""" % (linkto, interface, ip, mask, interface))
        else:
            veinterface = self._build_ve_interface(cname, interface)
            fd.write("""
lxc.network.type=veth
lxc.network.flags = up
lxc.network.link= br%s
lxc.network.name= %s
lxc.network.mtu = 1500
lxc.network.ipv4 = %%%%%s/%%%%calc_classe(%%%%%s)
lxc.network.veth.pair = %s
""" % (linkto, interface, ip, mask, veinterface))
        fd.close()
        file_path = join(self._make_root_path(cname), 'interfaces.d', interface)
        fd = open(file_path, 'w')
        fd.write("""
auto {0}
iface {0} inet static
address %%{1}
netmask %%{2}
broadcast %%{3}
""".format(interface, ip, mask, bcast))

    def _write_hosts_d(self, cname, name, ip, fname):
        file_path = join(self._make_root_path(cname), 'hosts.d', fname)
        fd = open(file_path, 'w')
        fd.write("%s  %s.%%%%nom_domaine_local %s\n" % (ip, cname, cname))
        fd.close()

    def lxc_install(self, container=None):
        if container == None:
            tcreate_containers_file = True
            fcontainers = file(config.containers_file, 'w')
            fcontainers.write('')
            fcontainers.close()
            tcontainer = self.containers
            fssh = open(config.container_ssh_config_file, 'w')
        else:
            if not self.containers.has_key(container):
                raise Exception('Le conteneur %s n\'existe pas'%container)
            tcreate_containers_file = False
            tcontainer = {}
            tcontainer[container] = self.containers[container]
        need_update = {}
        cont_var = load_container_var()
        br0 = cont_var['adresse_ip_br0']
        netmask = cont_var['adresse_netmask_br0']
        network = cont_var['adresse_network_br0'].rsplit('.', 1)[0]
        cidr = classes[netmask]
        for name, value in tcontainer.items():
            filename = join(self._make_root_path(name), 'config.d','disknods')
            if isfile(filename):
                unlink(filename)
        #generation du fichier write_fstab_config
        self.write_fstab_config()
        #vérification des points de montage (#3797)
        for conteneur in tcontainer.values():
            for disknod in conteneur['disknods']:
                if not ismount(disknod):
                    raise Exception("{0} doit être une partition".format(disknod))
        for name, value in tcontainer.items():
            cname = self.get_container_name(name, tcontainer)
            if not need_update.has_key(cname):
                need_update[cname] = True
            if config.VIRTDISABLED == False and cname != config.VIRTMASTER:
                if not tcontainer[cname].has_key('id'):
                    raise Exception("Le conteneur %s n'a pas d'attribut ID"%cname)
                containerid = tcontainer[cname]['id']
                if containerid == '':
                    raise Exception("Le conteneur %s n'a pas d'ID !" % name)
                assert 0 < int(containerid) < 255, "ID n'est pas un chiffre ou n'est pas compris entre 1 et 254"
                ip = '%s.%s' % (network, containerid)
                self.log.info('Installation du conteneur %s' % cname)
                cmd = ['/usr/share/eole/lxc_install.sh',
                       cname, str(name), '%s/%s' % (ip, cidr), br0] #, domain]
                if system_code(cmd) != 0:
                    raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                chroot = join(self._make_root_path(cname), config.VIRTBASE)
                #Reecrit le resolv.conf (#3001)
                ips_dns = self.variables['adresse_ip_dns'].get_value()
                if not eosfunc.is_empty(ips_dns):
                    fdns = file(join(chroot, 'etc/resolv.conf'), 'w')
                    for ip_dns in ips_dns:
                        fdns.write("nameserver {0}\n".format(ip_dns))
                    fdns.close()
                if tcreate_containers_file:
                    fcontainers = file(config.containers_file, 'a+')
                    fcontainers.write('container_path_%s="%s"\n' % (name, chroot))
                    fcontainers.write('container_ip_%s="%s"\n' % (name, ip))
                    fcontainers.write('container_name_%s="%s"\n' % (name, cname))
                    fcontainers.close()
                    fssh.write('Host %s\n'%name)
                    fssh.write('    Hostname=%s\n\n'%ip)
                self._write_hosts_d(cname, name, ip, 'host')
                fstab = []
                #ne demonte pas proc si pas monte
                mount_proc = False
                for package in value['packages']:
                    if not is_installed(package, container=name, context=False):
                        self.log.info('Installation de %s' % package)
                        #si non monte, monte les repertoires contenus dans le
                        #fichier fstab du conteneur
                        if fstab == []:
                            mtab = []
                            fmtab = file('/etc/mtab', 'r')
                            for line in fmtab.readlines():
                                origin, mount_point = line.split()[0:2]
                                mtab.append((origin, mount_point))
                            fmtab.close()
                            ffstab = file(join(self._make_root_path(cname), 'fstab'), 'r')
                            tfstab = []
                            for line in ffstab.readlines():
                                origin, mount_point = line.split()[0:2]
                                tfstab.append((origin, mount_point))
                            ffstab.close()
                            if isfile("/cdrom/cdromupgrade"):
                                if not isdir(join(self._make_root_path(cname), 'rootfs', 'cdrom')):
                                    makedirs(join(self._make_root_path(cname), 'rootfs', 'cdrom'))
                                tfstab.append(('/cdrom', join(self._make_root_path(cname), 'rootfs', 'cdrom')))
                            for origin, mount_point in tfstab:
                                if not isdir(origin):
                                    makedirs(origin)
                                if not (origin, mount_point) in mtab:
                                    cmd = ['mount', '--bind', origin, mount_point]
                                    if system_code(cmd) != 0:
                                        raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                                fstab.append(mount_point)
                        #monte /proc dans le conteneur si nécessaire
                        if not mount_proc:
                            # démontage si déjà monté (#3001)
                            system_out(['chroot', chroot, 'umount', '-n', '/proc'])
                            cmd = ['chroot', chroot, 'mount', '-n', '-t', 'proc',
                                        'proc', '/proc']
                            if system_code(cmd) != 0:
                                raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                            mount_proc = True
                        if not isfile(join(chroot, 'usr/sbin/invoke-rc.d.real')):
                            cmd = ['chroot', chroot, 'dpkg-divert',
                                   '--rename', '--divert',
                                   '/usr/sbin/invoke-rc.d.real',
                                   '/usr/sbin/invoke-rc.d']
                            if system_code(cmd) != 0:
                                raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                            wrapper = file(join(chroot, 'usr/sbin/invoke-rc.d'), 'w')
                            wrapper.write('#!/bin/sh\n')
                            wrapper.write('# EOLE Wrapper\n')
                            wrapper.write('echo "Wrapper"\n')
                            wrapper.close()
                            cmd = ['chmod', '755', join(chroot, 'usr/sbin/invoke-rc.d')]
                            if system_code(cmd) != 0:
                                raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))

                        if need_update[cname] == True:
                            cmd = ['/bin/cp', '-f', '/etc/apt/sources.list', join(chroot, 'etc/apt/sources.list')]
                            ret = system_out(cmd)
                            if ret[0] != 0:
                                raise Exception("Erreur d'exécution de %s\n%s" % (' '.join(cmd),
                                                                                  ret[2]))
                            cmd = ['chroot', chroot, '/usr/bin/apt-get', 'update']
                            ret = system_out(cmd)
                            if ret[0] != 0:
                                raise Exception("Erreur d'exécution de %s\n%s" % (' '.join(cmd),
                                                                                  ret[2]))
                            need_update[cname] = False
                        # installation forcée de eolebase-conteneur pour la gestion des dépendances (#1624)
                        cmd = ['chroot', chroot, '/usr/bin/apt-eole', 'install', package, 'eolebase-conteneur']
                        #Fix bug #1671
                        if package == 'antivirus-pkg':
                            cmd.append('clamd-eole-conteneur')
                        if system_code(cmd) != 0:
                            raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                disknods_file = open(join(self._make_root_path(cname), 'config.d','disknods'),'a')
                for disknod in value['disknods']:
                    if not ismount(disknod):
                        raise Exception("{0} doit être une partition".format(disknod))
                    try:
                        dev = getstatusoutput('mount | grep {0}'.format(disknod))[1].split()[0]
                        major_num = major(stat(dev).st_rdev)
                        minor_num = minor(stat(dev).st_rdev)
                    except:
                        raise Exception('Impossible de récupérer les propriétés du disknod {0}'.format(disknod))
                    disknods_file.write('lxc.cgroup.devices.allow = b {0}:{1} rwm\n'.format(major_num, minor_num))
                    if dev.startswith('/'):
                        dev = dev[1:]
                    dev_filename = join(self._make_root_path(cname), 'rootfs', dev)
                    dev_dir = dirname(dev_filename)
                    if not isdir(dev_dir):
                        makedirs(dev_dir)
                    if exists(dev_filename):
                        unlink(dev_filename)
                    mknod(dev_filename, 0660 | S_IFBLK, makedev(major_num, minor_num))
                disknods_file.close()
                if isfile(join(chroot, 'usr/sbin/invoke-rc.d.real')):
                    cmd = ['rm', '-rf', join(chroot, 'usr/sbin/invoke-rc.d')]
                    system_code(cmd)
                    cmd = ['chroot', chroot, 'dpkg-divert',
                           '--rename', '--remove',
                           '/usr/sbin/invoke-rc.d']
                    system_code(cmd)
                fstab.reverse()
                for mount_point in fstab:
                    cmd = ['umount', mount_point]
                    if system_code(cmd) != 0:
                        raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                #demonte /proc dans le conteneur
                if mount_proc:
                    system_code(['chroot', chroot, 'umount', '-n', '/proc'])

        if tcreate_containers_file:
            fcontainers = file(config.containers_file, 'a+')
            fcontainers.write('container_path_root=""\n')
            fcontainers.write('container_ip_root="127.0.0.1"\n')
            fcontainers.write('container_name_root="root"\n')
            fcontainers.close()
            fssh.close()

    def create_containers_file(self):
        """
        Génération du fichier /etc/eole/containers.conf en mode non conteneur
        """
        if config.VIRTDISABLED:
            #Regénère à chaque instance parce que on peut instancier un
            #eolebase puis un module #1018
            fcontainers = file(config.containers_file, 'w')
            for name in self.containers.keys():
                fcontainers.write('container_path_%s=""\n' % (name))
                fcontainers.write('container_ip_%s="127.0.0.1"\n' % (name))
                fcontainers.write('container_name_%s="%s"\n' % (name, config.VIRTMASTER))
            fcontainers.close()
            #Pas la peine de le regénérer
            if not isfile(config.network_containers_file):
                fcontainers = file(config.network_containers_file, 'w')
                fcontainers.write('adresse_ip_br0="127.0.0.1"\n')
                fcontainers.close()

    def install_pkg(self, container=None, context=True):
        """
        installation des paquets décrits dans les balises <package>
        """
        install_msg = 'Installation de %s (%s)'
        if container == None:
            tcontainer = self.containers
        else:
            if not self.containers.has_key(container):
                raise Exception('Le conteneur %s n\'existe pas'%container)
            tcontainer = {}
            tcontainer[container] = self.containers[container]

        if not config.VIRTDISABLED:
            # mode conteneur => gestion par conteneur
            for name, value in tcontainer.items():
                missing_pkg = []
                for package in value['packages']:
                    if not is_installed(package, container=name, context=context):
                        missing_pkg.append(package)
                if missing_pkg != []:
                    cmd = ['/usr/bin/apt-eole', 'update-conteneur', name]
                    if system_out(cmd)[0] != 0:
                        raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                    packages = ' '.join(missing_pkg)
                    self.log.info(install_msg % (packages, name))
                    print_cyan(install_msg % (packages, name))
                    cmd = ['/usr/bin/apt-eole', 'install-conteneur', name, packages]
                    if system_code(cmd) != 0:
                        raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
        else:
            # mode non conteneur => gestion globale
            missing_pkg = []
            for name, value in tcontainer.items():
                for package in value['packages']:
                    if not is_installed(package):
                        missing_pkg.append(package)
            if missing_pkg != []:
                cmd = ['/usr/bin/apt-get', 'update']
                if system_out(cmd)[0] != 0:
                    raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))
                packages = ' '.join(missing_pkg)
                self.log.info(install_msg % (packages, config.VIRTMASTER))
                print_cyan(install_msg % (packages, config.VIRTMASTER))
                cmd = ['/usr/bin/apt-eole', 'install', packages]
                if system_code(cmd) != 0:
                    raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))

    def lxc_postinstall(self, container=None):
        if config.VIRTDISABLED == True:
            return
        if container == None:
            tcontainer = self.containers
        else:
            if not self.containers.has_key(container):
                raise Exception('Le conteneur %s n\'existe pas' % container)
            tcontainer = {}
            tcontainer[container] = self.containers[container]
        #l'ensemble des groupes de conteneur
        cname = []
        for tname in tcontainer.keys():
            if tname != config.VIRTMASTER:
                cname.append(self.get_container_name(tname, tcontainer))
        cname = list(set(cname))
        for tname in cname:
            if config.VIRTDISABLED == False:
                cmd = ['/usr/share/eole/lxc_postinstall.sh', tname]
                if system_code(cmd) != 0:
                    raise Exception("Erreur d'exécution de %s" % ' '.join(cmd))

    def regroupe_services(self, tcontainer):
        """
        recalcule les services à activer en fonction des
        groupes de conteneurs réels (#1960)
        """
        services = {}
        for container, value in tcontainer.items():
            if config.VIRTDISABLED:
                gcontainer = config.VIRTMASTER
            else:
                gcontainer = self.get_container_name(container, tcontainer)
            if not services.has_key(gcontainer):
                services[gcontainer] = {'services': {}}
            for servicename, value in value['services'].items():
                if services[gcontainer]['services'].has_key(servicename):
                    #on garde la structure du 1er service chargé sauf pour active
                    if services[gcontainer]['services'][servicename]['active'] == False:
                        services[gcontainer]['services'][servicename]['active'] = value['active']
                else:
                    services[gcontainer]['services'][servicename] = value
        return services

    def instance_services(self, container=None):
        """
        instancie les services
        """
        if container == None:
            tcontainer = self.containers
        else:
            if not self.containers.has_key(container):
                raise Exception('Le conteneur %s n\'existe pas' % container)
            tcontainer = {}
            tcontainer[container] = self.containers[container]
        services = self.regroupe_services(tcontainer)
        for container, value in services.items():
            for servicename, value in value['services'].items():
                if value['active']:
                    # services à activer
                    update_rcd(service=str(servicename), action='set',
                            container=container, startlevel=str(value['startlevel']),
                            stoplevel=str(value['stoplevel']))
                else:
                    # services à désactiver
                    update_rcd(service=str(servicename), action='remove',
                            container=container)
                    service_out(str(servicename), 'stop', container)

    def restart_services(self, container=None, group=True, ead=False):
        """
        redemarrage des services
        """
        if container == None:
            tcontainer = self.containers
        else:
            if not self.containers.has_key(container):
                raise Exception('Le conteneur %s n\'existe pas' % container)
            tcontainer = {}
            tcontainer[container] = self.containers[container]
        services = self.regroupe_services(tcontainer)
        ordered_services = {}
        for name, value in services.items():
            container = config.VIRTMASTER
            if config.VIRTDISABLED == False:
                container = name
            for servicename, value in value['services'].items():
                if ead and servicename in config.EAD_SERVICES:
                    continue
                ordered_services[value['level']] = (servicename, container, value['active'])

        priority_ordered_serv = ordered_services.keys()
        if group == False:
            priority_ordered_serv.sort(reverse=True)
            # FIXME : on admet que networking est numéro 1
            priority_ordered_serv.pop()
            self. _restart_service(priority_ordered_serv, ordered_services, 'stop', container)
            service_code('networking', 'restart')
            action = 'start'
        else:
            action = 'restart'
        priority_ordered_serv.sort()
        self. _restart_service(priority_ordered_serv, ordered_services, action, container)

    def _restart_service(self, priority_ordered_serv, ordered_services, action, container):
        for level in priority_ordered_serv:
            servicename = ordered_services[level][0]
            container = ordered_services[level][1]
            active = ordered_services[level][2]
            if not active:
                continue
            else:
                service_code(servicename, action, container)

    def write_init_service(self):
        """
        sauvegarde des types de démarrage dans le fichier init_service.conf
        """
        finit = file(config.init_services_file, 'w')
        dict_services = {}
        for container, values in self.containers.items():
            for servicename, value in values['services'].items():
                if dict_services.has_key(servicename):
                    containers = dict_services[servicename][2]
                    if value['active'] and container not in containers:
                        containers.append(container)
                else:
                    if value['active']:
                        containers = [container]
                    else:
                        containers = []
                dict_services[servicename] = (value['method'], value['pty'], containers)
        json.dump(dict_services, finit)
        finit.close()

    def load_values(self, store_file=config.configeol, force=False, check=True):
        """initialise les valeurs des variables depuis un fichier cfg"""
        # vérification des groupes de variables ?
        try:
            ok_files = self.store.read(store_file)
            if type(store_file) != list and store_file not in ok_files:
                return False
        except:
            return False
        return self._load_values(force=force, check=check)

    def _load_values(self, force=False, check=False):
        # liste des variables non présentes dans config.eol
        not_filled = []
        for varname in self.variables.keys():
            #si varname est dans virt_not_load c'est qu'il ne faut pas
            #recharger la valeur de fichier config.eol
            if varname in self.virt_not_load:
                continue
            var = self.variables[varname]
            if self.store.has_section(varname):
                var.val = self.store.get(varname, 'val')
                # ConfigParser renvoie une chaine, il faut un objet liste
                if type(var.val) != list:
                    var.val = eval(var.val)
                var.valprec = self.store.get(varname, 'valprec')
                if type(var.valprec) != list:
                    var.valprec = eval(var.valprec)
                var.valdefault = self.store.get(varname, 'valdefault')
                if type(var.valdefault) != list:
                    var.valdefault = eval(var.valdefault)
            else:
                # éventuellement une seule valeur par défaut si nouvelle slave
                # il vaut mieux la laisser à vide et la récupérer par get_value
                # self.variables[varname].val = self.variables[varname].valeole
                self.variables[varname].val = []
                not_filled.append(varname)

            # mise à jour des variables esclaves pour les groupes
            if varname in self.groups:
                var.update_slaves()
        for varname, var in self.variables.items():
            # on recalcule toutes les contraintes de type fill avant de lancer
            # les vérifications
            if var.val == [] or var.val == ['']:
                var.init_fills()
            var.init_auto()
        for varname in not_filled:
            # charge la valeur par défaut si la variable n'est pas renseignée
            # dans le config.eol.
            self.variables[varname].val = self.variables[varname].get_value()
        for varname in self.variables.keys():
            # on vérifie toutes les variables
            if force == False:
                self.check_conditions(varname)
        for varname, var in self.variables.items():
            # boucle spécifique une fois que tout a été calculé
            if check == True:
                self.check_value(var, force)
        return True

    def reset_defaults(self):
        """recharge les valeurs par défaut du dictionnaire"""
        # on commence par remettre en place les valeurs 'eole' partout
        # pour éviter les problèmes avec les fonctions 'fills' des variables groupées
        for key in self.variables:
            self.variables[key].val = self.variables[key].valeole
        # calcul des valeurs réelles (fills, auto, ...)
        for key in self.variables:
            self.variables[key].val = self.variables[key].get_value(default=True)

    def check_conditions(self, varname, force=False, force_hide=False):
        """vérifie les contraintes imposées à la variable
        @param force: si force==True, on ne tient pas compte des valeurs obligatoires
        (utile pour saisir les valeurs par défaut d'une variante ou d'un module)"""
        # calcul des variables cachées
        if self.constraints.has_key(varname):
            if not self.variables.has_key(varname):
                raise TypeEoleError("variable de condition \"%s\" inconnue" % varname)
            vals = self.variables[varname].get_value()
            # XXX hack that fixes 3351
            if vals == []:
                vals = ['']
            for val in vals:
                for func in self.constraints[varname]:
                    active = True
                    # si la cible est connue, on l'active/desactive
                    try:
                        params = [{'value':val}]
                        params.extend(func[7])
                        if not (func[0] == 'obligatoire' and force == True):
                            self.variables[varname].eval_func(func[0], params)
                    except DependencyError:
                        # on désactive les variables liées
                        active = False
                    if force_hide == True:
                        active = False
                    for family_target in func[1]:
                        if family_target in self.families:
                            self.families[family_target]['hidden'] = not active
                    for target in func[2]:
                        if target in self.variables:
                            if force_hide == False and active == False:
                                self.check_conditions(target, force=force, force_hide=True)
                            else:
                                self.check_conditions(target, force=force)
                            self.variables[target].set_active(active)
                    for file_target in func[3]:
                        if file_target in self.files:
                            self.files[file_target]['hidden'] = not active
                    for filelist_target in func[4]:
                        if self.filelists.has_key(filelist_target):
                            for tmpl in self.filelists[filelist_target]:
                                self.files[tmpl]['hidden'] = not active
                    for servicelist_target in func[5]:
                        #if self.servicelists.has_key(service_target):
                        for name in self.containers.keys():
                            for servicename in self.containers[name]['services']:
                                if self.containers[name]['services'][servicename]['servicelist'] == servicelist_target:
                                    self.containers[name]['services'][servicename]['active'] = active
                    for interfacelist_target in func[6]:
                        for name in self.containers.keys():
                            for interfacename in self.containers[name]['interfaces']:
                                if self.containers[name]['interfaces'][interfacename]['interfacelist'] == interfacelist_target:
                                    self.containers[name]['interfaces'][interfacename]['active'] = active
        # calcul des variables automatiques
        if self.auto_vars.has_key(varname):
            for autovar in self.auto_vars[varname]:
                var = self.variables[autovar]
                try:
                    val = var.get_calculated_value(auto=True)
                except DependencyError:
                    # la valeur auto n'est pas active
                    pass
                    # var.val = var.valprec
                else:
                    var.set_value(val)

    def check_value(self, var, force=False):
        force_var = force
        if var.hidden == True:
            force_var = True
        else:
            for family in self.families.values():
                if var.name in family['vars']:
                    if family['hidden'] == True:
                        # famille hidden : on ne vérifie pas 'obligatoire'
                        force_var = True
                        break
        var.check_value(var.val, force_var)


    def save_values(self, store_file=config.configeol, force=False):
        """sauvegarde les valeurs des variables"""
        for varname in self.variables.keys():
            var = self.variables[varname]
            # la variable est hidden (ou une famille la contenant)
            # on prend la valeur par défaut si pas de valeur
            if var.val == [] or var.val == ['']:
                var.init_fills()
                if var.auto != []:
                    try:
                        val = var.get_calculated_value(auto=True)
                    except DependencyError:
                        pass
                    else:
                        var.val = val
            self.check_value(var, force)

            # vérification de la valeur
            # passage des chaines unicode en str
            for list_val in [var.val, var.valprec, var.valdefault]:
                for ind in range(len(list_val)):
                    if type(list_val[ind]) == unicode:
                        list_val[ind] = list_val[ind].encode(config.charset)

            if not self.store.has_section(varname):
                self.store.add_section(varname)
            self.store.set(varname, 'val', var.val)
            self.store.set(varname, 'valprec', var.valprec)
            self.store.set(varname, 'valdefault', var.valdefault)
        # sauvegarde du fichier
        try:
            file_conf = file(store_file, 'w')
            self.store.write(file_conf)
            file_conf.close()
        except:
            return False

        return True

    def todict(self, hide_vars=False):
        dictionnaire = {}
        masters = {}
        for maitre, esclaves in self.groups.items():
            for esclave in esclaves:
                masters[esclave] = maitre
        for family in self.families.values():
            for var in family['vars']:
                if hide_vars and (family['hidden'] == True or self.variables[var].hidden == True):
                    continue
                else:
                    # valeurs multi-valuee : liste, valeur simple : chaine !
                    var = stringify(var)
                    dictionnaire[var] = self.get_value(var)
                    if var in masters:
                        if self.variables[masters[var]].multi:
                            continue
                    if self.variables[var].multi:
                        continue
                    if len(dictionnaire[var]) == 1:
                        dictionnaire[var] = str(dictionnaire[var][0])
                    elif len(dictionnaire[var]) == 0:
                        dictionnaire[var] = ""
        return dictionnaire

    def get_files(self):
        """
        @return: la liste des fichiers cibles
        """
        return self.files

    def get_containers(self):
        """
        @return: la liste des fichiers cibles
        """
        return self.containers

    def get_variables(self):
        """
        @return: dictionnaire des variables et leur valeur
        """
        return self.variables

    def get_families(self):
        """
        @return: dictionnaire des familles et fonctions associées
        """
        return self.families

    def get_normal_families(self):
        """
        @return: dictionnaire des familles NON EXPERT et fonctions associées
        """
        families = {}
        # FIXME : optimisation ?
        for family in self.normal_families:
            # on ne garde que les clés valides
            families[family] = self.families[family]
        return families

    def get_groups(self):
        """
        @return: dictionnaire des groupes
        """
        return self.groups

    def get_variable(self, name):
        """
        @return: la valeur de la variable name
        """
        return self.variables[name]

    def get_constraints(self):
        """
        @return: liste des contraintes
        """
        return self.constraints

    def get_services(self, container=None):
        """
        @return: liste des services activés
        """
        if container == None:
            tcontainer = self.containers
        else:
            if not self.containers.has_key(container):
                raise Exception('Le conteneur %s n\'existe pas' % container)
            tcontainer = {}
            tcontainer[container] = self.containers[container]
        gservices = self.regroupe_services(tcontainer)
        services = set()
        for cname, value in gservices.items():
            for servicename, value in value['services'].items():
                if value['active'] and  value['method'] != 'apache':
                    services.add((servicename, cname))
        return services

if __name__ == "__main__":
    pass
