# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
#  Fonctions de calcul et validation des variables Creole
#
###########################################################################
from commands import getstatusoutput
import subprocess
from pyeole2.process import system_out
from creole2.config import func_dir, containers_file, network_containers_file, \
eole_module, VIRTDISABLED, containers_default_network
from creole2.error import TypeEoleError, DependencyError, FileNotFound, \
 ConfigError
from creole2.utils import classes, gen_random
#from creole2 import ifconfig
from creole2 import wpkg_secrets
import imp

# modules utilitaires à rendre accessible dans les fonctions
import os, sys, re, glob, time, hashlib, random, socket

# type paramètre
class Param:
    """paramètre pour les fonctions de contrainte"""
    def __init__(self, value, name=None, typ=None):
        self.value = value
        self.name = name
        self.type = typ

def load_container_var():
    """
    retourne le contenu de containers.conf et network_containers.conf
    sous forme de dictionnaire
    """
    ret = {}
    if os.path.isfile(network_containers_file):
        execfile(network_containers_file, {}, ret)
    if os.path.isfile(containers_file):
        execfile(containers_file, {}, ret)
    return ret

#############################
## Fonctions de vérification
#############################

def valid_ip(data):
    """fonction de validation de la saisie d'une adresse ip"""

    # variable non obligatoire
    if data == "":
        return True

    # découpage de l'ip
    ip = data.split('.')
    if len(ip) != 4:
        raise TypeEoleError("mauvaise syntaxe d'adresse ip")

    # vérification des nombres
    for part in ip:
        try:
            num = int(part)
        except ValueError:
            raise TypeEoleError("l'ip doit être constituée de 4 nombres séparés par des points")
        else:
            if not (0 <= num < 256):
                raise TypeEoleError("les nombres doivent etre compris entre 0 et 255")

    # on n'a pas rencontré d'erreur
    return True

def valid_intervalle(data):
    """ fonction de validation d'une plage d'ip """
    if data == "":
        return True

    liste = data.split()
    if len(liste) != 2:
        raise TypeEoleError("vous devez spécifier 2 adresses ip")
    else:
        # on vérifie la syntaxe des ips
        for ip in liste:
            valid_ip(ip)
    return True

def valid_network(data, ip=None):
    """fonction de validation de la saisie d'un réseau"""
    if data == "":
        return True

    # on utilise la fonction de test d'adresse ip pour la syntaxe
    valid_ip(data)
    if ip is not None:
        # on effectue les tests spécifiques au réseau (fonction de l'ip ?)
        ip_parts = ip.split(".")
        network_parts = data.split(".")
        for i in range(4):
            if (int(network_parts[i]) != 0) and (network_parts[i] != ip_parts[i]):
                raise TypeEoleError("le réseau spécifié ne correspond pas à l'ip "+ip)

    # aucune erreur rencontrée
    return True

def valid_netmask(data):
    """fonction de validation de la saisie d'un masque réseau"""
    if data == "":
        return True

    # on utilise la fonction de test d'adresse ip pour la syntaxe
    valid_ip(data)
    # on effectue les tests spécifiques aux masques réseaux
    netmask = []
    for part in data.split("."):
        netmask.append(int(part))
    error = 0
    mask_len = 0
    done = 0
    for part in netmask:
        bitmask = 0x80
        for bit in range(8):
            if not done:
                if part & bitmask:
                    mask_len = mask_len + 1
                else:
                    done = 1
            else:
                if part & bitmask:
                    error = 1
            bitmask = bitmask >> 1

    if error == 1:
        raise TypeEoleError("syntaxe de masque réseau invalide")
    # pas d'erreur rencontrée
    return True

def valid_booleen(data, type_bool="alpha"):
    """fonction de validation d'une réponse booléenne
       type_bool : alpha ou num (O/N ou 0/1)
    """
    if data == "":
        return True

    if (data[0].upper() != 'O') and (data[0].upper() != 'N'):
        raise TypeEoleError("vous devez repondre par 'O' ou 'N'")
    # pas d'erreurs détectées
    return True

def valid_lower(data):
    """
    validation d'une valeur en minuscules
    """
    if data != data.lower():
        raise TypeEoleError("la valeur doit être en minuscules")
    return True

def valid_enum(data, liste=['oui','non','Oui','Non','OUI','NON','o','n','O','N'], checkval="True"):
    """fonction de validation d'une réponse en fonction
    d'une liste de valeur possibles
    """
    if data == "":
        return True

    try:
        if type(liste) != list:
            liste = eval(liste)
    except:
        raise TypeEoleError("la liste des valeurs possibles n'est pas valide")
    if data not in liste and checkval != "False":
        raise TypeEoleError("repondez par un des choix suivants : "+str(liste))
    return True

def valid_regexp(data, exp_reg, err_msg="la syntaxe est incorrecte"):
    """fonction de validation d'une saisie par expression régulière"""
    if data == "":
        return True
    match = re.match(exp_reg, data)
    if match is None:
        raise TypeEoleError(err_msg)
    else:
        return True

def valid_entier(data, mini=None, maxi=None):
    """ fonction de validation d'un nombre (intervalle optionnel)"""
    if data == "":
        return True
    try:
        value = int(data)
    except ValueError:
        raise TypeEoleError("vous devez saisir un nombre")
    else:
        if mini is not None and value < int(mini):
            raise TypeEoleError("donnez un entier supérieur à "+str(mini))
        if maxi is not None and value > int(maxi):
            raise TypeEoleError("donnez un entier inférieur à "+str(maxi))
        return True

def valid_yesno(data):
    """Yes ou No pour les fichiers de config"""
    if data not in ["Yes", "No"]:
        raise TypeEoleError("la valeur doit être Yes ou No")
    return True

def valid_system(data, cmd, result, *args):
    """ test d'une commande système """
    if data == "":
        return True

    cmd_string = cmd
    for arg in args:
        cmd_string += " "+arg

    # exécution de la fonction
    code_retour = os.system(cmd_string)

    if code_retour == result:
        return True
    else:
        raise TypeEoleError("la commande "+cmd_string+" a echoué")

def valid_differ(data, value):
    """ test d'une valeur indésirable """
    if data == value:
        raise TypeEoleError("la valeur doit être différente de %s"%value)
    else:
        return True

def valid_chaine(data, forbidden_chars="[' ']"):
    """
    Recherche de caractères interdits dans une chaine
    """
    forbidden_chars = eval(forbidden_chars)
    for char in forbidden_chars:
        if char in data:
            raise TypeEoleError("Le caractère \"%s\" est interdit" % (char))
    return True

def valid_name(data):
    """
    nom valide
    """
    valid_lower(data)
    valid_chaine(data, forbidden_chars="['@', ' ', '_', '+']")
    return True

def valid_domaine(data):
    """
    nom de domaine valide
    """
    valid_name(data)
    valid_regexp(data, '.+\..+', "Le nom de domaine nécessite une extension")
    return True

def valid_alias(data):
    """
    alias apache valide
    """
    valid_name(data)
    valid_regexp(data, '^/', "Les alias doivent débuter par un \"/\"")
    return True

def valid_phpmyadmin(data, apache):
    """ phpmyadmin dépend d'apache """
    if data == 'oui' and apache == 'non':
        raise TypeEoleError("L'activation de PhpMyAdmin nécessite celle d'Apache")
    return True

def valid_addr_sso(data, sso_actif, revprox_actif="non", revprox_sso="", revprox_domainname="", *args):
    """vérifie les conflits entre sso local et reverse proxy"""
    if data in ['localhost', '127.0.0.1']:
        raise TypeEoleError("Le servive sso doit écouter sur une interface autre que localhost")
    if revprox_sso and revprox_actif == 'oui':
        if data in args and sso_actif == 'oui':
            raise TypeEoleError("Le service sso local est en conflit avec le reverse proxy, utilisez %s comme adresse sso" % (revprox_domainname or revprox_sso))
    return True

def verif_cas_active(data, var_name, full_cas):
    """ vérifie l'activation du full_cas
    """
    if data == 'oui' and full_cas == 'non':
        raise TypeEoleError("Veuillez activer l'\"Utilisation du service sso pour les applications de votre serveur Scribe\" avant d'activer %s" % var_name)
    return True


def obligatoire(data):
    """ fonction qui vérifie que les données ne sont pas vides
    """
    if data == [] or data == "" or data[0].strip() == "":
        raise TypeEoleError("ce paramètre est obligatoire")
    else:
        return True

def is_empty(data):
    if str(data) in ['', '""', "''", "[]", "['']", '[""]']:
        return True
    return False

################################################################
## Fonctions de calcul de dépendance entre variables et groupes
################################################################

def hidden_if_not_in(val, *args):
    """ on cache si la valeur n'est pas présente dans celles proposées """
    if val not in args:
        raise DependencyError("""%s n'est pas dans %s""" % (val, args))

def hidden_if_in(val, *args):
    """ on cache si la valeur est présente dans celles proposées """
    if val in args:
        raise DependencyError("""%s est dans %s""" % (val, args))

# TODO : fonctions du genre range() ??

#############################################
## Fonctions de saisie et calcul des valeurs
#############################################

def get_devices():
    cmd = ['/sbin/ip', '-4', '-o', 'address', 'show', 'up', 'primary']
    code, stdout, stderr = system_out(cmd)

    if code != 0:
        raise Exception('erreur à la commande {0} : {1}'.format(' '.join(cmd), stderr))

    devs = {}
    for line in stdout.splitlines():
        dev = line.strip().split()
        if dev[2] != 'inet':
            continue
        #get aliases
        devname = dev[-1]
        if '/' in dev[3]:
            ip, netmask = dev[3].split('/')
        else:
            ip = dev[3]
            netmask = '32'
        if dev[4] == 'brd':
            broadcast = dev[5]
        else:
            broadcast = ''
        devs[devname] = (ip, calc_netmask(netmask), broadcast)

    return devs

def calc_network(ip, netmask):
    """ calcule le réseau par défaut à partir d'un masque et d'une ip """
    # si les ip et netmask viennent du dictionnaire, on les récupère
    if ip == "" or netmask == "":
        return ""
    broadcast, network = calc_adresses(ip, netmask)

    return network

def calc_broadcast(ip, netmask):
    """ calcule l'adresse de broadcast par défaut à partir d'un masque et d'une ip """
    # si les ip et netmask viennent du dictionnaire, on les récupère
    if ip == "" or netmask == "":
        return ""
    broadcast, network = calc_adresses(ip, netmask)

    return broadcast

def create_ip(ip_list):
    """ création d'une adresse ip à partir d'une liste"""
    template = """%i.%i.%i.%i"""
    return template % tuple(ip_list)

def create_ip_list(ip):
    """ création d'une liste à partir d'une adresse ip"""
    ipa, ipb, ipc, ipd = ip.split('.')
    return [int(ipa), int(ipb), int(ipc), int(ipd)]

def calc_adresses(ip, mask):
    """calcul l'adresse de broadcast à partir de l'ip et du netmask
    """
    mask_list = create_ip_list(mask)
    ip_list = create_ip_list(ip)
    broadcast_list = []
    network_list = []
    for i in range(4):
        #pour l'adresse réseau c'est simple
        network_list.append(int(ip_list[i]) & int(mask_list[i]))
        #broadcast_part=255 si mask[i]=255
        for j in range(256):
            if mask_list[i]&j == 0:
                broadcast_part = j
        #calcul du masque spécial
        if broadcast_part != 255:
            broadcast_part += mask_list[i]&ip_list[i]
        broadcast_list.append(broadcast_part)
    broadcast = create_ip(broadcast_list)
    network = create_ip(network_list)
    return broadcast, network

def calc_classe(netmask):
    """calcule la classe du réseau à partir du netmask"""

    if classes.has_key(netmask):
        return(classes[netmask])
    else:
        return(netmask)

def calc_netmask(plage):
    """calcule le reseau a partir de sa classe"""
    if plage in classes.values():
        for mask, plaj in classes.items():
            if plaj == plage:
                return mask
    else:
        return plage

def calc_pg_cache_size():
    """
    Calcule la taille du cache postgreql en fonction de la RAM
    """
    procmem = open('/proc/meminfo')
    meminfo = procmem.read().split('\n')
    procmem.close()
    for line in meminfo:
        if line.startswith('MemTotal'):
            mem_tot = float(line.split()[1])
            break
    mem_cache = int( (mem_tot * 0.50) / 8 )
    return str(mem_cache)

def calc_mysql_innodb_size():
    """
    Calcule la taille du cache mysql en fonction de la RAM
    """
    procmem = open('/proc/meminfo')
    meminfo = procmem.read().split('\n')
    procmem.close()
    for line in meminfo:
        if line.startswith('MemTotal'):
            mem_tot = float(line.split()[1])
            break
    mem_cache = int( (mem_tot * 0.72) /1024)
    return str(mem_cache)

def calc_val(valeur):
    """ valeur par défaut "en dur" """
    return(valeur)

def concat(*args, **kwargs):
    """ concatène les chaînes passées en paramètre """
    if '' in args:
        return ""
    sortedkeys = kwargs.keys()
    sortedkeys.sort()
    sortedkwvalues = []
    for key in sortedkeys:
        if kwargs[key] == '':
            return ""
        sortedkwvalues.append(kwargs[key])
    return "".join(args)+''.join(sortedkwvalues)

def calc_container(mode_container, container_dict, mode_zephir='non'):
    """ retourne une variable de conteneur """
    container_dict = eval(container_dict)
    if mode_zephir == 'non':
        if not os.path.isfile(containers_file):
            return ""
        try:
            container_var = load_container_var()
            if container_dict['name'] == 'br0':
                variable = 'adresse_ip_%s' % container_dict['name']
            else:
                variable = 'container_ip_%s' % container_dict['name']
            return container_var[variable]
        except KeyError:
            raise DependencyError("""Impossible de calculer la variable de conteneur %s""" % variable)
    else:
        if mode_container == 'non':
            return '127.0.0.1'
        else:
            return '%s.%s' % (containers_default_network, container_dict['id'])

def calc_multi_val(*args):
    """ valeur par défaut "en dur" avec un nombre non défini de paramètres
    Attention, une valeur vide passée en paramètre à cette fonction force
    le renvoi d'une liste vide ([])
    Les doublons sont supprimés
    """
    res = []
    for arg in args:
        if arg == '':
            return []
        # gestion des valeurs multi
        try:
            if arg.startswith("["):
                ev_args = eval(arg)
                assert type(ev_args) == list
                for ev_arg in ev_args:
                    if ev_arg != "" and ev_arg not in res:
                        res.append(ev_arg)
            else:
                if arg != '':
                    res.append(arg)
        except:
            if arg != '':
                res.append(arg)
    return(res)

def calc_multi_condition(param, match, mismatch, **conditions):
    """param: réponse testée sur toutes les conditions. Si les réponses des conditions sont différentes
    mettre une liste avec chacune des conditions a tester
    match: valeur de retour si toutes les conditions sont validées
    mismatch: valeur de retour si au moins une des conditions n'est pas validée
    conditions: liste des conditions a tester.
    exemple:
        <auto name='calc_multi_condition' target='test_activer_bacula_dir'>
            <param>['oui', 'non']</param>
            <param type='eole' name='condition_1'>activer_bacula</param>
            <param type='eole' name='condition_2'>activer_bacula_dir</param>
            <param name='match'>oui</param>
            <param name='mismatch'>non</param>
        </auto>
    """
    if param.startswith('['):
        param = eval(param)
        assert type(param) == list
        if len(param) != len(conditions):
            raise ValueError('Nombre de conditions et de params différent dans calc_multi_condition')
    for num in range(0, len(conditions)):
        if not conditions.has_key('condition_%s'%(num+1)):
            raise ValueError('Condition %s calc_multi_condition inconnue'%num+1)
        value = conditions['condition_%s'%(num+1)]
        if type(param) != list:
            if value != param:
                return mismatch
        else:
            if value != param[num]:
                return mismatch
    return match


def calc_binary_and(valeur1, valeur2):
    """
        Calcule un 'et' binaire :
        valeur1 & valeur2
    """
    if valeur1 == 'oui' and valeur2 == 'oui':
        return('oui')
    else:
        return('non')

def calc_group_dn(groupe, num_etab, acad=''):
    """
    """
    acad = acad.split('.')[0]
    return "cn="+groupe+",ou=local,ou=groupes,ou="+num_etab+",ou="+acad+",ou=education,o=gouv,c=fr"

def calc_libelle_annuaire(ldap_host, nom_machine, nom_domaine_local):
    """
    calcule un libellé pour l'annuaire local
    """
    if ldap_host == '':
        raise DependencyError
    if ldap_host in ['127.0.0.1', 'localhost']:
        return "Annuaire de %s.%s" % (nom_machine, nom_domaine_local)
    else:
        try:
            hostname = socket.gethostbyaddr(ldap_host)[0]
        except:
            hostname = ldap_host
        return "Annuaire de %s" % hostname

def get_home_path():
    """Récupère le chemin d'accès au homedir
    (/home par défaut, /partages sur OpenVZ)
    """
    return '/home'

def calc_last_interface(nombre_int):
    return 'eth'+str(int(nombre_int)-1)


#################################################
## Fonctions de calcul des valeurs automatiques
#################################################

def calc_network_auto(ip, mask):
    """calcul l'adresse de broadcast à partir de l'ip et du netmask
    """
    mask_list = create_ip_list(mask)
    ip_list = create_ip_list(ip)
    network_list = []
    for i in range(4):
        #pour l'adresse réseau c'est simple
        network_list.append(int(ip_list[i]) & int(mask_list[i]))
    network = create_ip(network_list)
    return network

def server_mem():
    """renvoit la mémoire du serveur
    """
    procmem = open('/proc/meminfo')
    meminfo = procmem.read().split('\n')
    procmem.close()
    for line in meminfo:
        if line.startswith('MemTotal'):
            mem_tot = float(line.split()[1])
            break
    return str(mem_tot)

def kernel_version():
    """ renvoit la version du kernel
    """
    return_code, output = getstatusoutput('uname -r')
    if return_code == 0:
        return str(output)
    else:
        return ''

def auto_copy_val(value):
    """
        Renvoie la valeur value
        utilisé pour forcer la valeur d'une variable
    """
    return(value)

def auto_eth(nom_carte, **kw):
    """ renvoit l'adresse de la carte nom_carte
    """
    if kw['parametre'].lower() == 'pppoe':
        nom_carte = 'ppp0'
    if kw['condition'].lower() == kw['parametre'].lower() or nom_carte == 'ppp0':
        #eths = ifconfig.ifconfig()
        #print eths.getInterfaceList()
        #if nom_carte in eths.getInterfaceList():
        #    eth = eths.getAddr(nom_carte)
        eths = get_devices()
        if 'br'+nom_carte in eths.keys():
            nom_carte = 'br'+nom_carte
        return str(eths.get(nom_carte, ('', '', ''))[0])
    else:
        raise DependencyError

def auto_netmask(nom_carte, **kw):
    """ renvoit le netmask pour la carte nom_carte
    """
    if kw['parametre'].lower() == 'pppoe':
        nom_carte = 'ppp0'
    if kw['condition'].lower() == kw['parametre'].lower() or nom_carte == 'ppp0':
        #eths = ifconfig.ifconfig()
        #if nom_carte in eths.getInterfaceList():
        #    netmask = eths.getNetmask(nom_carte)
        #else:
        #    return ''
        eths = get_devices()
        return str(eths.get(nom_carte, ('', '', ''))[1])
    else:
        raise DependencyError

def auto_network(nom_carte, **kw):
    """ renvoit le network pour la carte nom_carte
    """
    if kw['parametre'].lower() == 'pppoe':
        nom_carte = 'ppp0'
    if kw['condition'].lower() == kw['parametre'].lower() or nom_carte == 'ppp0':
        #eths = ifconfig.ifconfig()
        #if nom_carte in eths.getInterfaceList():
        #    ip = eths.getAddr(nom_carte)
        #    netmask = eths.getNetmask(nom_carte)
        #    network = calc_network_auto(ip, netmask)
        #else:
        #    return ''
        eths = get_devices()
        ip, netmask, broadcast = eths.get(nom_carte, ('', '', ''))
        if ip != '' and netmask != '':
            network = calc_network_auto(ip, netmask)
        else:
            network = ''
        return str(network)
    else:
        raise DependencyError

def auto_ip_ldap(activer_client_ldap, **kw):
    """ renvoie l'adresse du serveur ldap local
    pas de valeur renvoyée si on choisit un ldap distant
    """
    if activer_client_ldap != 'local':
        raise DependencyError
    else:
        # récupération de l'adresse du service local
        return kw['ip_annuaire']

def auto_broadcast(nom_carte, **kw):
    """ renvoit le broadcast pour la carte nom_carte
    """
    if kw['parametre'].lower() == 'pppoe':
        nom_carte = 'ppp0'
    if kw['condition'].lower() == kw['parametre'].lower() or nom_carte == 'ppp0':
        #eths = ifconfig.ifconfig()
        #if nom_carte in eths.getInterfaceList():
        #    broadcast = eths.getBroadcast(nom_carte)
        #else:
        #    return ''
        eths = get_devices()
        return str(eths.get(nom_carte, ('', '', ''))[2])
    else:
        raise DependencyError

def auto_defaultroute(**kw):
    """ renvoit la route par défaut
    """
    cmd = "ip route list scope global"
    cmd = cmd.split()
    process = subprocess.Popen(args=cmd, stdout=subprocess.PIPE,
                               stderr=subprocess.PIPE, shell=False)
    code = process.wait()
    if code != 0:
        return False
    defaultroute = process.stdout.read()
    defaultroute = re.findall('default\s+via\s+([\d\.]+)\s+dev\s+(\w+)\s+metric\s+', defaultroute)
    if len(defaultroute) == 0:
        cmd = "ip route list"
        cmd = cmd.split()
        process = subprocess.Popen(args=cmd, stdout=subprocess.PIPE,
                                   stderr=subprocess.PIPE, shell=False)
        code = process.wait()
        if code != 0:
            return False
        defaultroute = process.stdout.read()
        defaultroute = re.findall('default\s+dev\s+(\w+)\s+', defaultroute)
        if len(defaultroute) == 0:
            return [['0.0.0.0', 'eth0']]
        else:
            defaultroute = [['0.0.0.0', defaultroute[0]]]
    return defaultroute

def auto_defaultgw_ip(method):
    """ renvoit la route par défaut
    """
    if method == "statique":
        raise DependencyError
    #DHCP
    ip = auto_defaultroute()
    return ip[0][0]

def auto_defaultgw_int(method):
    """ renvoit la route par défaut
    """
    if method == "statique":
        return 'eth0'
    elif method == "pppoe":
        return 'ppp0'
    else:
        #DHCP
        rint = auto_defaultroute()
        return rint[0][1]

def auto_defaultroute_ip(method, calc_ip_gw):
    """ renvoit la route par défaut
    """
    if method == "statique" or calc_ip_gw == "non":
        raise DependencyError
    #DHCP
    ip = auto_defaultroute()
    return ip[0][0]

def auto_defaultroute_int(method, calc_ip_gw):
    """ renvoit la route par défaut
    """

    if method == "statique" or calc_ip_gw == "non":
        raise DependencyError
    if method == "pppoe":
        return 'ppp0'
    #DHCP
    rint = auto_defaultroute()
    return rint[0][1]

def auto_module():
    return eole_module

def mode_conteneur():
    #si VIRTDISABLED est à True, retourne non, sinon oui
    return {True: "non", False: 'oui'}.get(VIRTDISABLED)

####################################################
# fonctions diverses accessibles dans les templates
####################################################

def list_files(rep, wildcards='*', stripext=False, basename=True, default=[], exception=''):
    """
    Liste des fichiers dans un répertoire
    @rep: répertoire de recherche
    @wildcards: expression régulière sur les fichiers à rechercher
    @stripext: supprimer l'extension des fichiers
    @basename: ne conserver que le nom court des fichiers
    @default: liste à renvoyer dans le cas où aucun fichier n'est trouvé
    @exception: les fichiers contenant cette chaîne sont ignorés
    """
    res = []
    if os.path.isdir(rep):
        for fic in glob.glob(rep+'/%s' % wildcards):
            if basename:
                fic = os.path.basename(fic)
            if stripext:
                fic = os.path.splitext(fic)[0]
            if exception:
                if exception in fic:
                    continue
            res.append(fic)
    if res == []:
        res = default
    res.sort()
    return res

def is_file(filename):
    """ filtre permettant de verifier l'existence
    d'un fichier avant de l'inclure par exemple """
    return os.path.isfile(filename)

def to_list(lst, sep=' '):
    """retourne une liste concatenee de donnees de la liste"""
    return sep.join(lst)

def purge_list(lst):
    """supprime les doublons dans une liste"""
    res = []
    for elt in lst:
        if elt not in res:
            res.append(elt)
    return res

def to_iso(data):
    """encode une chaine en iso"""
    try:
        return unicode(data, "UTF-8").encode("ISO-8859-1")
    except:
        return data

def to_sql(data):
    """echape les caractères ' et " d'une chaine"""
    try:
        return data.replace("'", "\\\'").replace("\"", "\\\"")
    except:
        return data

def is_installed(package, chroot=None):
    """ vérifie si un paquet est installé ou pas"""
    """FIXME: obsolete, utiliser plutot pyeole.process.is_installed"""
    cmd = ["/usr/bin/dpkg", "-l", package.strip()]
    if chroot != None:
        cmd = ['chroot', chroot] + cmd
    process = subprocess.Popen(args=cmd,
              stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=False)
    code = process.wait()
    if code != 0:
        # paquet non installé
        return False
    dpkg_info = process.stdout.read()
    pkg_status = dpkg_info.strip().split('\n')[-1].split()
    if pkg_status[0].lower() in ['pn', 'un']:
        return False
    return True

def installed_version(package):
    """renvoie la version d'un paquet"""
    process = subprocess.Popen(args=["dpkg", "-l", package.strip()],
              stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=False)
    code = process.wait()
    if code != 0:
        # paquet non installé
        return ''
    dpkg_info = process.stdout.read()
    pkg_status = dpkg_info.strip().split('\n')[-1].split()
    if pkg_status[0].lower() == 'pn':
        # ancien paquet purgé (status = pn)
        return ''
    return pkg_status[2]

def custom_join(multivar, separator=' '):
    """ mise à plat d'une variable multiple"""
    return separator.join([i.value for i in multivar])

def is_ip(data):
    """vérifie si data décrit une adresse IP"""
    if data == '':
        return False
    try:
        return valid_ip(data)
    except:
        return False

def certs_nuauth():
    """ liste des certificats pour nuauth """
    namecert = ''
    nuauth_tls_key = ''
    rep = '/etc/ipsec.d'
    repdefault = '/etc/nufw/certs/'
    wildcards = '*.pem'
    if os.path.isdir(rep):
        if len(glob.glob(rep+'/%s' % wildcards)) > 0:
            namecert = os.path.basename(glob.glob(rep+'/%s' % wildcards)[0])
            nuauth_tls_key = os.path.join(rep, 'private', 'priv%s' % namecert)
            if not os.path.isfile(nuauth_tls_key):
                nuauth_tls_key = ''
            else:
                nuauth_tls_cert = os.path.join(rep, namecert)
                if not os.path.isfile(nuauth_tls_cert):
                    nuauth_tls_key = ''
                else:
                    nuauth_tls_cacert = os.path.join(rep, 'cacerts', 'CertifCa.pem')
                    if not os.path.isfile(nuauth_tls_cacert):
                        nuauth_tls_key = 'none'
    if nuauth_tls_key == '':
        nuauth_tls_key = os.path.join(repdefault,'nuauth-eole.key')
        nuauth_tls_cert = os.path.join(repdefault,'nuauth-eole.crt')
        nuauth_tls_cacert = '/etc/ssl/certs/ca.crt'
    res = [nuauth_tls_key, nuauth_tls_cert, nuauth_tls_cacert]
    return res

def key_nuauth():
    """retourne la clef privee pour nuauth"""
    return certs_nuauth()[0]

def cert_nuauth():
    """retourne le certificat pour nuauth"""
    return certs_nuauth()[1]

def cacert_nuauth():
    """retourne l'authorité de certification pour nuauth"""
    return certs_nuauth()[2]

def test_vz(master=False):
    """
    master = True : on teste si c'est un serveur hôte
    master = False : on teste si c'est un serveur virtualisé
    """
    if master:
        package = 'eolevz'
        openvz = os.system('dpkg -l %s 2>/dev/null|grep -q "^ii"' % package)
        return openvz == 0
    else:
        if os.path.exists('/etc/sysconfig/eole/eolevz'):
            return True
        else:
            return False

def upper(string):
    """ texte en majuscules """
    return string.upper()

def lower(string):
    """ texte en minuscules """
    return string.lower()

def wpkg_crypt(string):
    return wpkg_secrets.wcrypt(string)

def wpkg_decrypt(string):
    return wpkg_secrets.wdecrypt(string)

def get_file_list(string):
    return glob.glob(string)

def escape_regexp(string):
    return string.replace('.', '\.')

#########################################################################
# Chargement de fonctions supplémentaires depuis /usr/share/creole/funcs
#########################################################################

def load_funcs(reload_mod=True):
    dict_module = globals()
    for mod_file in glob.glob('%s/*.py' % func_dir):
        try:
            if os.path.basename(mod_file) != '__init__.py':
                name = os.path.splitext(os.path.basename(mod_file))[0]
                mod_perso = imp.load_source(name, mod_file)
                for attr in dir(mod_perso):
                    if not attr.startswith('__'):
                        if attr not in dict_module:
                            dict_module[attr] = getattr(mod_perso, attr)
                        elif reload_mod == False:
                            # on n'écrase pas les fonctions déjà présentes !
                            print "%s - déjà défini : %s" % (mod_file, attr)

        except Exception, e:
            print "erreur lors du chargement des fonctions personnalisées dans %s" % mod_file
            print str(e)

# chargement initial du module
load_funcs(False)

def main():
    """fonction de test  de la bibliothèque"""
    try:
        valid_netmask(sys.argv[1])
        print "ok"
    except Exception, msg:
        print "erreur : %s" % str(msg)

if __name__ == "__main__":
    main()
