# -*- coding: UTF-8 -*-
"""
Parseur LXML des fichiers XML de collecte des variables EOLE
"""
from lxml import etree
from creole2.error import ConfigError
from creole2.utils import string_to_bool #, get_text_node
from creole2.config import VIRTMASTER

service_level = 0
file_level = 0


def parse_xml_file(filename):
    """
    @param filename: nom du fichier xml source
    @return: structure de données permettant de créer les objets Eole
    """
    try:
        document = etree.iterparse(filename, events=('end',), tag='creole')
        return _parse_root_node(document)
    except Exception, err:
        import traceback
        traceback.print_exc()
        raise ConfigError, "Erreur lors du parsing du fichier %s : %s" % (filename, str(err))

def parse_string(xml_string):
    """
    @param xml_string: dictionnaire xml sous forme de chaîne
    @return: structure de données permettant de créer les objets Eole
    """
    try:
        root_node = etree.fromstring(xml_string)
        document = etree.iterwalk(root_node, events=('end',), tag='creole')
        return _parse_root_node(document)
    except Exception, err:
        import traceback
        traceback.print_exc()
        raise ConfigError, "Erreur lors du parsing : %s" % str(err)


def _parse_root_node(document):
    """
    @param document: le noeud XML racine
    """
    for _, first_node in document:
        root_node = first_node

    variables = parse_variables(root_node.find('variables'))
    families, order = parse_families(root_node)

    # balise <files> (données sur le maître)
    file_node = root_node.findall('files')
    softwares = {}
    if file_node != []:
        if len(file_node) != 1:
            raise Exception("Erreur plusieurs balises <files> par dictionnaire")
        file_root_node = file_node[0]
        files = parse_files(file_root_node)
        file_packages = parse_packages(file_root_node)
        file_services = parse_services(file_root_node)
        softwares[VIRTMASTER] = [files, file_packages, file_services, {}, '1', '', '']

    # balise <containers> (données dans les conteneurs)
    containers_node = root_node.findall('containers')
    if containers_node != []:
        for container in containers_node:
            softwares.update(parse_containers(container))

    ## gestion des contraintes
    constraints = parse_constraints(root_node)

    # gestion des groupes de variables
    groups = parse_groups(root_node)

    # gestion de l'aide
    helps = parse_help(root_node)

    # gestion des séparateurs
    separators = parse_separators(root_node)

    return softwares, variables, families, order, constraints, groups, helps, separators


def _get_boolean_attr(node, attr_name, default=False):
    """
    Gestion spécifique pour les attributs booléens
    Ils sont à False par défaut
    """
    val = node.get(attr_name)
    if default:
        return str(val).lower() != 'false'
    else:
        return str(val).lower() == 'true'


def _get_optionnal(node, attr_name):
    """
    Valeur d'un attribut optionnel
    renvoie une chaine vide au lieu de None
    """
    val = node.get(attr_name)
    if val is None:
        val = ''
    return val


def _parse_value(varnode, attr='value'):
    """
    récupération des valeurs d'une variable
    """
    res = []
    for val in varnode.findall(attr):
        # FIX for <value></value> !
        if val.text is not None:
            res.append(val.text)
        else:
            res.append('')
    return res


def parse_variables(var_node):
    """
    traitement des variables
    @param var_node: noeud <variables>
    """
    result = {}
    for var in var_node.getiterator('variable'):
        hidden = _get_boolean_attr(var, 'hidden')
        multi = _get_boolean_attr(var, 'multi')
        redefine = _get_boolean_attr(var, 'redefine')
        mandatory = _get_boolean_attr(var, 'mandatory')
        remove_check = _get_boolean_attr(var, 'remove_check')
        exists = _get_boolean_attr(var, 'exists', default=True)
        mode = _get_optionnal(var, 'mode')
        if mode != 'expert':
            mode = 'normal'
        name = var.attrib['name']
        value = _parse_value(var)
        typ = _get_optionnal(var, 'type')
        desc = _get_optionnal(var, 'description')
        result[name] = dict(value=value,
                            type=typ,
                            description=desc,
                            hidden=hidden,
                            multi=multi,
                            auto='',
                            redefine=redefine,
                            exists=exists,
                            mode=mode,
                            mandatory=mandatory,
                            remove_check=remove_check
                            )
    return result


def _get_vars_name(family_node):
    """
    récupération de la liste des variables associées à une famille
    @param family_node: noeud XML de la famille courante
    """
    varsname = []
    for var in family_node.getiterator('variable'):
        varsname.append(var.attrib['name'])
    return varsname


def parse_families(var_node):
    """
    traitement des familles
    @param var_node: noeud <variables>
    """
    result = {}
    order = []
    for family in var_node.findall('variables/family'): #: getiterator('family'):
        family_name = family.attrib['name']
        order.append(family_name)
        hidden = _get_boolean_attr(family, 'hidden')
        # FIXME: mode='' était admis avec domparser
        mode = _get_optionnal(family, 'mode')
        result[family_name] = {'hidden':hidden,
                               'mode':mode,
                               'vars':_get_vars_name(family)
                              }
    return result, order


def parse_files(file_node):
    """
    Parsing des balises <file>
    @file_node : noeud xml
    """
    global file_level
    result = {}
    for fic in file_node.getiterator('file'):
        file_level += 1

        result[fic.attrib['name']] = {'hidden':False,
                                      'source': _get_optionnal(fic,'source'),
                                      'mode': _get_optionnal(fic, 'mode'),
                                      'owner': _get_optionnal(fic, 'owner'),
                                      'group': _get_optionnal(fic, 'group'),
                                      'mkdir': _get_boolean_attr(fic, 'mkdir'),
                                      'filelist': _get_optionnal(fic, 'filelist'),
                                      'rm': _get_boolean_attr(fic, 'rm'),
                                      'del_comment': _get_optionnal(fic, 'del_comment'),
                                      'container_only': _get_boolean_attr(fic, 'container_only'),
                                      'level': file_level
                                      }
    return result


def parse_packages(package_node):
    """
    Traitement de balises <package>
    """
    return _parse_value(package_node, 'package')


def parse_services(service_node):
    """
    Traitement des balises <service>
    """
    global service_level
    result = {}
    for serv in service_node.getiterator('service'):
        service_level += 1
        method = serv.get('method')
        if not method:
            method = 'service'
        redefine = _get_boolean_attr(serv, 'redefine')
        hidden = _get_boolean_attr(serv, 'hidden')
        # FIXME : cas inverse de d'habitude !
        try:
            pty = string_to_bool(serv.get('pty'))
        except (TypeError, ValueError):
            pty = True
        # FIXME : cas encore différent !
        try:
            #True : seulement en mode conteneur
            #False : seulement en mode non conteneur
            #None : dans les deux modes
            in_container = string_to_bool(serv.get('in_container'))
        except (TypeError, ValueError):
            in_container = None
        #result[str(get_text_node(serv))] = {'servicelist':serv.getAttribute('servicelist'),
        result[serv.text] = {'servicelist': _get_optionnal(serv, 'servicelist'),
                             'startlevel': _get_optionnal(serv, 'startlevel'),
                             'stoplevel': _get_optionnal(serv, 'stoplevel'),
                             'method': method,
                             'pty': pty,
                             'level': service_level,
                             'in_container': in_container,
                             'redefine': redefine,
                             'hidden': hidden,
                             }
    return result


def parse_disknod(disknod_node):
    """
    Traitement de balises <disknod>
    """
    return _parse_value(disknod_node, 'disknod')


def parse_interfaces(interface_node):
    """
    Traitement des balises <interface>
    """
    result = {}
    for interface in interface_node.getiterator('interface'):
        method = interface.get('method')
        if not method:
            method = 'macvlan'
        if method not in ['macvlan', 'bridge']:
            raise Exception('method pour une interface doit être macvlan ou bridge et pas {0}'.format(method))
        result[interface.text] = {'interfacelist': _get_optionnal(interface, 'interfacelist'),
                                  'linkto': _get_optionnal(interface, 'linkto'),
                                  'ip': _get_optionnal(interface, 'ip'),
                                  'mask': _get_optionnal(interface, 'mask'),
                                  'bcast': _get_optionnal(interface, 'bcast'),
                                  'method': method
                                 }
    return result


def parse_containers(containers_node):
    """
    Traitement des conteneurs
    @param containers_node: noeud XML des conteneurs
    """
    result = {}
    for container_node in containers_node.getiterator('container'):
        name = container_node.attrib['name']
        containerid = _get_optionnal(container_node, 'id')
        groupid = _get_optionnal(container_node, 'group')
        if name == VIRTMASTER:
            raise Exception("Le nom '%s' est interdit pour une balise <container>" % VIRTMASTER)
        if name in result:
            raise Exception("Le nom %s doit etre unique par dictionnaire" % name)

        file_nodes = parse_files(container_node)
        packages = parse_packages(container_node)
        services = parse_services(container_node)
        interfaces = parse_interfaces(container_node)
        disknods = parse_disknod(container_node)
        result[name] = [file_nodes, packages, services, interfaces, containerid, groupid, disknods]
    return result


def parse_constraints(node):
    """
    @param node: node des contraintes
    """
    constraints = {'checks' : parse_funcs(node,'check'),
                   'fills' : parse_funcs(node,'fill'),
                   'auto' : parse_funcs(node,'auto'),
                   'conditions' : parse_conditions(node)
                  }
    return constraints


def _parse_param(param_node):
    """
    traitement des paramètres d'une fonction
    """
    return {'name'  : _get_optionnal(param_node, 'name'),
            'type'  : _get_optionnal(param_node, 'type'),
            'value' : param_node.text,
            'optional' : _get_optionnal(param_node, 'optional'),
            'hidden' : _get_optionnal(param_node, 'hidden'),
            }


def parse_funcs(node, func_type):
    """
    @param node: node des fonctions
    @param func_type: TagName of the functions to find
    """
    # fonctions de vérification
    funcs = {}
    #for func in node.getiterator(func_type):
    for func in node.findall('constraints/%s' % func_type):
        # lecture des paramètres
        # XXX (adim): est-ce que le fait que params ne soit pas réinitialisé
        #             à chaque tour de boucle "for target in targets" est un bug ?
        params = []
        targets = _parse_value(func, 'target') # [ get_text_node(t) for t in func.getElementsByTagName('target') ]
        if not targets:
            targets = [_get_optionnal(func, 'target')]
        for target in targets:
            if target is not None:
                for param in func.getiterator('param'):
                    params.append(_parse_param(param))
                funcs.setdefault(target, []).append([func.attrib['name'],
                                                     params])
    return funcs


def parse_conditions(node):
    """
    @param node: node des fonctions
    """
    # fonctions de vérification
    funcs = {}
    for func in node.getiterator('condition'):
        # lecture des paramètres
        targets = []
        family_targets = []
        file_targets = []
        filelist_targets = []
        servicelist_targets = []
        interfacelist_targets = []
        # paramètres de la fonction
        params = [_parse_param(param)
                  for param in func.getiterator('param')]
        # cibles de la dépendance
        for target in func.getiterator('target'):
            ttype = target.get('type')
            if ttype == 'family':
                family_targets.append(target.text)
            elif ttype == 'file':
                file_targets.append(target.text)
            elif ttype == 'filelist':
                filelist_targets.append(target.text)
            elif ttype == 'servicelist':
                servicelist_targets.append(target.text)
            elif ttype == 'interfacelist':
                interfacelist_targets.append(target.text)
            else:
                # sinon, c'est une variable EOLE
                targets.append(target.text)
        funcdef = [func.attrib['name'], family_targets, targets,
                   file_targets, filelist_targets, servicelist_targets, interfacelist_targets, params]
        funcs.setdefault(_get_optionnal(func, 'source'), []).append(funcdef)
    return funcs


def parse_groups(node):
    """
    Traitement des groupes de variables
    """
    result = {}
    for group in node.findall('constraints/group'):
        slaves = _parse_value(group, 'slave')
        result[group.attrib['master']] = slaves
    return result


def parse_help(node):
    """
    Traitement de l'aide
    """
    var_help = {}
    for var in node.findall('help/variable'):
        name = var.attrib['name']
        try:
            var_help[name] = var.text.strip()
        except AttributeError:
            raise Exception("Aide invalide pour la variable {0}".format(name))
    fam_help = {}
    for var in node.findall('help/family'):
        name = var.attrib['name']
        try:
            fam_help[name] = var.text.strip()
        except AttributeError:
            raise Exception("Aide invalide pour la famille {0}".format(name))
    return {'variables':var_help, 'families': fam_help}


def parse_separators(node):
    """dictionnaire des séparateurs, format {'variable':'text'}
    variable : nom de la première variable après le sépateur"""
    var_sep = {}
    for var in node.findall('variables/separators/separator'):
        if not var.text:
            libelle = ''
        else:
            libelle = var.text.strip()
        var_sep[var.attrib['name']] = (libelle, _get_boolean_attr(var, 'never_hidden'))
    return var_sep

