# -*- coding: UTF-8 -*-

"""
Objets Eole
"""

import sys
from pyeole2.odict import OrderedDict

from creole2 import eosfunc
from creole2.error import TypeEoleError, DependencyError
from creole2.config import charset
from creole2.utils import encode_list

def type_factory(mime, name, **args):
    """
    @param mime: le type mime du type recherché
    @param name: le nom de la variable eole à créer
    @return: un objet type correspondant au mimetype
    """
    if not isinstance(mime, basestring):
        raise TypeEoleError("mauvais type mime '%s' pour la variable %s " % (str(mime), str(name)))
    if not mime.strip():
        raise TypeEoleError("type de variable vide pour la variable : %s" % str(name))

    mime = mime.lower()
    if mime not in REGISTERED_TYPES:
        raise TypeEoleError("type de variable inconnu [%s] pour la variable %s" % (str(mime), str(name)))
    return REGISTERED_TYPES[mime](name, **args)

REGISTERED_TYPES = {}

# objet de base Eole
class autoregistrable(type):
    def __new__(mcs, name, bases, classdict):
        mime = classdict.get('mime')
        cls = type.__new__(mcs, name, bases, classdict)
        if mime:
            REGISTERED_TYPES[mime] = cls
        return cls


class MultiVariable:
    """
    >>> var = MultiVariable(12, dict(netmask='truc', network='muche'))
    >>> var
    12
    >>> var.netmask
    'truc'
    >>> var.network
    'muche'
    """
    def __init__(self, value, slavedict):
        self.value = value
        self.__dict__.update(slavedict)

    def __repr__(self):
        return str(self.value)

    def __eq__(self, other):
        # pour gérer les comparaisons Multivalué/chaine
        if hasattr(other,'value'):
            return self.value == other.value
        else:
            return self.value == other

class EoleVar:
    __metaclass__ = autoregistrable

    def __init__(self, name, context={}, description=None, val=None, valprec=[], valdefault=[], valeole=[], hidden=False, multi=False, slave=False, checks=[], fills=[], auto=[], mode='normal', linter=False):
        """
        contient normalement les tests sur l'objet
        @param name: nom de la variable
        @param description: libellé de la variable
        @param val: valeur utilisée
        @param valprec: valeur précédente
        @param valdefault: valeur par défaut régionale
        @param valeole: valeur par défaut d'origine
        @param checks: liste des fonction(s) de vérification supplémentaire
        @param fills: liste des fonction(s) de calcul de valeur par défaut
        @param auto: variable automatique ?
        @param linter: spécifique à CreoleLint, ne pas toucher
        """
        self.eole_dico = context
        if context != {}:
            self.context = context.variables
        else:
            self.context = {}
        self.auto = auto
        if val == None:
            self.val = []
            # self.val = valeole
        else:
            self.val = val
        #valprec doit être à la valeur au chargement
        if valprec == [] and self.val != []:
            valprec = self.val
        self.valprec = valprec
        self.valdefault = encode_list(valdefault)
        self.valeole = encode_list(valeole)
        # autres paramètres
        self.name = name
        self.description = description
        self.hidden = hidden
        self.multi = multi
        self.slave = slave
        self.num_values = 0
        self.mode = mode
        self.linter = linter
        # initialisation de la liste des fonctions
        if not type(checks) == list or not type(fills) == list:
            raise TypeEoleError("Erreur à l'initiatisation des fonctions")
        self.checks = checks
        self.fills = fills

    def init_fills(self):
        if self.fills != []:
            self.val = self.get_calculated_value()

    def init_auto(self):
        if self.auto != []:
            try:
                self.val = self.get_calculated_value(auto=True)
            except DependencyError:
                # on ne doit pas utiliser la valeur auto
                pass
    def __eq__(self, other):
        return self.name == other.name and \
               self.val == other.val and \
               self.valprec == other.valprec and \
               self.valdefault == other.valdefault and \
               self.valeole == other.valeole and \
               self.description == other.description

    def set_active(self, active):
        self.hidden = not active

    def set_value(self, vals, default=False, force = False):
        """affectation d'une valeur à la variable
        @param vals : valeurs
        @param default: assignation de la valeur par défaut si True
        """
        # si la valeur n'est pas dans un liste, on en crée une
        if type(vals) != list:
            vals = [vals]
        # vérification des valeurs (lève une exception si erreur)
        for val in vals:
            self.check_value([val], force)
        if not default :
            if self.val != vals:
                # inutile d'écraser si idem
                self.valprec = []
                self.valprec.extend(self.val)
                self.val = vals
        else:
            self.valdefault = vals
        # mise à jour des variables esclaves si besoin
        self.update_slaves()

    def check_value(self, vals, force = False):
        # parcours des couples (fonction, arguments)
        for couple in self.checks:
            try:
                # cas particulier : valeurs obligatoires
                if couple[0] == 'obligatoire':
                    if force == True or self.hidden == True:
                        # en mode forcé, on ignore les variables obligatoires
                        continue
                    eosfunc.obligatoire(vals)
                # les variables sont de type liste
                # on doit appliquer la fonction sur chacune
                for val in vals:
                    valeurs = [{'value':val}]
                    if couple[1] != []:
                        valeurs.extend(couple[1])
                    # évaluation fonction avec les valeurs en 1er argument
                    self.eval_func(couple[0], valeurs)
            except TypeEoleError, err:
                # on précise le nom de la variable dans le message
                raise TypeEoleError("%s : %s" % (str(self.name), str(err)))

    def eval_func(self, function, params, index=0):
        # parcours des arguments
        arguments = ''
        for param in params:
            # gestion des keyword args
            if param.has_key('name'):
                if param['name'] != '':
                    arguments += '%s=' % str(param['name'])
            value = param['value']
            # gestion des références aux variables eole
            if param.has_key('type'):
                if param['type'] == 'eole':
                    if not self.context.has_key(param['value']):
                        if param['optional'] != 'True':
                            if self.linter == False:
                                raise TypeEoleError("la variable %s n'est pas connue (paramètre de condition)" % str(param['value']))
                        else:
                            # variable optionelle non présente, on passe à l'argument suivant
                            continue
                    else:
                        # si le param hidden='False' est definit, si la variable est caché, elle n'est pas ajoutée
                        # si la variable n'est pas caché, le processus continue
                        if param['hidden'] == 'False' and self.context[param['value']].hidden == True:
                            continue
                        vals = self.context[param['value']].get_value()
                        if len(vals) == 0:
                            value = ''
                        else:
                            if self.is_linked(param['value']):
                                # la variable est groupée
                                value = vals[index]
                            else:
                                if self.context[param['value']].multi == True:
                                    # si variable multi et non groupée, on prend toutes les valeurs
                                    value = vals
                                else:
                                    # sinon on prend toujours la première
                                    value = vals[0]

                elif param['type'] == 'python':
                    try:
                        value = eval(value, globals(), locals())
                    except:
                        if param['optional'] != 'True':
                            print "erreur lors de l'évaluation d'un paramètre (%s)" % str(self.name)
                            raise ValueError(str(self.name))
                        else:
                            # paramètre ignoré
                            continue
                elif param['type'] == 'container':
                    if self.eole_dico.containers.has_key(param['value']):
                        if self.eole_dico.containers[param['value']].has_key('group'):
                            group = self.eole_dico.containers[param['value']]['group']
                            value = self.eole_dico.containers[group]
                        else:
                            value = self.eole_dico.containers[param['value']]
                    elif param['value'] == 'br0':
                        value = {'name':'br0', 'id':1}
                    else:
                        value = {}
            # valeur du paramètre
            if type(value) is unicode:
                value = value.encode(charset)
            arguments += '"%s", ' % str(value).replace('"', '\\"')

        # les fonctions sont forcément dans eosfunc
        # on fait un eval car on n'a que le nom de la fonction
        if type(function) is unicode:
            function = function.encode(charset)
        if type(arguments) is unicode:
            arguments = arguments.encode(charset)
        try:
            return eval( 'eosfunc.%s(%s)' % (function, arguments) )
        except DependencyError, err:
            raise DependencyError(str(err))
        except Exception, err:
            err_line = sys.exc_info()[2].tb_lineno
            raise TypeEoleError("""%s - erreur à l'exécution de %s (ligne %s): %s""" % (str(self.name), str(function), err_line, str(err)))

    def is_linked(self, varname):
        for master, slaves in self.eole_dico.groups.items():
            if self.name == master or self.name in slaves:
                if varname == master or varname in slaves:
                    return True
        return False

    def get_value(self, default=False):
        """retourne la valeur actuelle
        @param default: force le retour de la valeur par défaut
        @return: valeur actuelle ou valeur par défaut
        (si pas de valeur par défaut, retour de la valeur eole)
        """
        # si la variable est automatique, on tente de renvoyer la valeur calculée si il y a
        if self.auto != []:
            try:
                return self.get_calculated_value(auto=True)
            except DependencyError:
                # on ne doit pas utiliser la valeur auto
                pass
        # cas d'une variable 'slave', on vérifie si toutes les valeurs sont présentes
        slave_err = False
        if self.slave != False:
            slave_err = len(self.eole_dico.variables[self.slave].get_value()) != len(self.val)
        #if not default and self.val not in ([], ['']) and not (self.val in ([], ['']) and self.slave != False):
        if not default and self.val not in ([], ['']) and not slave_err:
            return self.val
        if self.valdefault != []:
            if self.slave:
                if len(self.valdefault) <= 1:
                    vals = []
                    num_master = len(self.eole_dico.variables[self.slave].get_value())
                    num_values = len(self.val)
                    # on renvoie toujours au moins 1 valeur, même si la variable master n'en a pas
                    for index in range(num_master or 1):
                        # on ajoute le nombre de valeurs manquantes
                        if index < num_values and self.val[index] != '' and not default:
                            vals.append(self.val[index])
                        else:
                            vals.extend(self.valdefault)
                else:
                    vals = self.valdefault
                return vals
            else:
                return self.valdefault
        else:
            if len(self.fills) > 0:
                calc_vals = self.get_calculated_value()
                if self.slave != False and not default:
                    # slaves: ajout des valeurs manquantes si default non demandé
                    vals = []
                    num_master = len(self.eole_dico.variables[self.slave].get_value())
                    num_values = len(self.val)
                    for index in range(num_master or 1):
                        if index < num_values and self.val[index] != '' and not default:
                            vals.append(self.val[index])
                        else:
                            vals.append(calc_vals[index])
                    return vals
                return calc_vals
            if self.valeole != [] or self.slave:
                if self.slave:
                    if len(self.valeole) <= 1:
                        vals = []
                        # variable esclave : on renvoie toujours autant de valeurs que la variable maitre
                        # si une seule valeur par défaut
                        num_master = len(self.eole_dico.variables[self.slave].get_value())
                        num_values = len(self.val)
                        # on renvoie toujours au moins 1 valeur, même si la variable master n'en a pas
                        for index in range(num_master or 1):
                            # on ajoute le nombre de valeurs manquantes
                            if index < num_values and self.val[index] != '' and not default:
                                vals.append(self.val[index])
                            else:
                                vals.extend(self.valeole or [''])
                    else:
                        # sinon on prend la valeur telle quelle
                        vals = self.valeole
                    return vals
                else:
                    return self.valeole
            else:
                return []

    def get_final_value(self):
        """
        :return: valeur directement utilisable pour Cheetah
        """
        value = self.get_value()
        if value in ([], ['']):
            if not self.multi and self.slave == False:
                return ''
            else:
                return value

        if self.multi:
            return list(self)

        else:
            # On utilise getitem pour gérer la transformation en
            # MultiVariable quand nécessaire
            return self.get_final_value_at_index(0)

    def get_final_value_at_index(self, index):
        """On renvoie la une valeur directement utilisable
        par Cheetah (avec transformation en MultiVariable si nécessaire)
        """
        value = self.get_value()[index]
        slaves = self.get_slaves()
        if slaves:
            slavedict = {}
            for varname, values in slaves.items():
                slavedict[varname] = values[index]
            return MultiVariable(value, slavedict)
        else:
            return value

    def get_prec_value(self):
        """retourne la valeur précédente
        """
        return self.valprec


    def get_calculated_value(self, index=None, auto=False):
        vals = []
        if auto == True:
            funcs = self.auto
        else:
            funcs = self.fills

        for fill in funcs:
            # calcul de la valeur
            func = fill[0]
            params = fill[1]
            # on regarde si cette variable est liée à self pour récupérer
            # le bon index de valeur
            grouped = 0
            for param in params:
                if param['type'] == 'eole':
                    if self.is_linked(param['value']):
                        grouped = len(self.context[param['value']].val)
                        break
            if grouped != 0:
                vals = []
                for i in range(grouped):
                    vals.append(self.eval_func(func, params, i))
            else:
                vals = self.eval_func(func, params)
                if type(vals) != list:
                    vals = [vals]
        if self.slave and len(vals) == 1:
            # si on est dans une variable esclave avec un fill renvoyant une seule valeur,
            # on retourne la valeur autant de fois que nécessaire (ref #3647)
            num_values = len(self.eole_dico.variables[self.slave].get_value())
            if num_values > 1:
                vals_multi = []
                for index in range(num_values):
                    vals_multi.extend(vals)
                return vals_multi
        return vals

    def update_slaves(self):
        """
        calcul du nombre de valeurs requises pour les esclaves
        """
        if hasattr(self.eole_dico, 'groups'):
            if self.name in self.eole_dico.groups:
                #num_values = len(self.val)
                num_values = len(self.get_value())
                for slave in self.eole_dico.groups[self.name]:
                    self.eole_dico.variables[slave].num_values = num_values

    def get_slaves(self):
        if self.multi:
            self_values = self.get_value()
        slaves = OrderedDict()
        # on teste si self est une variable maître
        if self.name in self.eole_dico.groups:
            # on récupère le nom des variables esclaves
            for slave in self.eole_dico.groups[self.name]:
                # on récupère les valeurs de la variable esclave
                slave_values = self.eole_dico.variables[slave].get_value()
                if slave_values == [] and len(self.val) == 1:
                    # variable facultative dans l'esclave fixes #1794
                    slaves[slave] = ['']
                else:
                    slaves[slave] = slave_values
                    # si c'est multi-valué, il **faut** que les variables esclaves
                    # aient le même nombre de valeurs que la variable maître
                    # cas particulier : liste vide pour la variable maitre, mais des valeurs
                    #                   par défaut existent dans les variables  esclaves.
                    if self.multi and len(self_values) != 0:
                        assert len(slave_values) == len(self_values), \
                               "La variable esclave '%s' doit avoir le meme nombre d'elements" \
                               " que la variable maitre '%s'" % (slave, self.name)
        return slaves

    def __getitem__(self, index):
        """On rend la variable indexable"""
        assert self.multi, "impossible d'utiliser la notation indexee si la " \
               "variable n'est pas multi-valuee"
        return self.get_final_value_at_index(index)

    def __iter__(self):
        """On rend la variable itérable pour pouvoir faire:
        for ip in iplist:
            print ip.network
            print ip.netmask
            print ip
        """
        assert self.multi, "impossible d'iterer sur une variable non multiple"
        #tout le bloc du dessous était dans le IF
        #corrige #1197
        #if self.hidden == False:
        self_values = self.get_value()
        slaves = self.get_slaves()
        for index, value in enumerate(self_values):
            slavedict = {}
            for varname, values in slaves.items():
                slavedict[varname] = values[index]
            # on rajoute un attribut "index" pour que l'indice de la valeur
            # soit accessible directement sur la variable, e.g. :
            # %for $ip in $liste_ips
            #  $ip.index # <=== ici !
            # %end for
            slavedict['index'] = index
            yield MultiVariable(value, slavedict)

# définition des types de variables Creole

class TypedVar(EoleVar):
    checkers = []

    def __init__(self, name, **kwargs):
        EoleVar.__init__(self, name, **kwargs)
        self.checks = self.checkers + self.checks

class String(TypedVar):
    mime = 'string'

class Ip(TypedVar):
    mime = 'ip'
    checkers = [ ['valid_ip', []], ]

class Netmask(TypedVar):
    mime = 'netmask'
    checkers = [ ['valid_netmask', []], ]

class Number(TypedVar):
    mime = 'number'
    checkers = [ ['valid_entier', []], ]

class Boolean(TypedVar):
    mime = 'boolean'
    checkers = [ ['valid_booleen', []], ]

class OuiNon(TypedVar):
    mime = 'oui/non'
    checkers = [ ['valid_enum', [{'name': '', 'value':u"['oui','non']"}]] ]

class OnOff(TypedVar):
    mime = 'on/off'
    checkers = [ ['valid_enum', [{'name': '', 'value':u"['on','off']"}]] ]
