# -*- coding: UTF-8 -*-
###########################################################################
# Eole NG - 2007
# Copyright Pole de Competence Eole  (Ministere Education - Academie Dijon)
# Licence CeCill  cf /root/LicenceEole.txt
# eole@ac-dijon.fr
#
# creolewrap.py
#
# classe abstraite permettant de gérer les configurations
# creole1 et creole2 dans zephir
#
###########################################################################

from os import close
import os, traceback
from copy import copy
from tempfile import mkstemp
from glob import glob
from collections import OrderedDict

from zephir.config import charset, PATH_ZEPHIR, PATH_TEMP, PATH_MODULES, DISTRIBS
from zephir.config import NOBODY_GID
from zephir.config import NOBODY_UID

def check_path_temp():
    if not os.path.isdir(PATH_TEMP):
        os.makedirs(PATH_TEMP)
        if os.getuid() != NOBODY_UID:
            # si lancé par le backend, on force le propriétaire
            # (répertoire utilisé également par l'application web)
            os.chown(PATH_TEMP, NOBODY_UID, NOBODY_GID)

# import creole1
import dico, base64, cjson, ConfigParser

# creole2
from creole2.error import TypeEoleError
from creole2 import eosfunc, cfgparser

# creole3 (eole2.4/tiramisu)
from creole import upgrade
from creole.loader import config_save_values, load_config_store, \
config_get_values, config_load_store, modes_level, CreoleVarLoader, load_store
from creole.var_loader import convert_value
from tiramisu.value import Multi
from tiramisu.option import ChoiceOption, OptionDescription, BoolOption, IntOption
from tiramisu.error import PropertiesOptionError
from tiramisu.setting import owners
from tiramisu.config import Config

# HACK pour simuler les variables de conteneurs dans Zéphir (attention, toutes les ips sont à 127.0.0.1)
eosfunc.containers_file = '/usr/share/zephir/dictionnaires/fake_containers.conf'
cfgparser.config.containers_file = '/usr/share/zephir/dictionnaires/fake_containers.conf'
EDict = cfgparser.EoleDict

def check_hidden(properties):
    """regarde si une famille/variable creole 3 est cachée
    """
    hidden_properties = set(['hidden', 'disabled'])
    if hidden_properties.intersection(properties):
        return True
    return False

def get_load_errors(config):
    """détecte les erreurs stockées à l'import d'un fichier de configuration
    """
    load_errors = {}
    setting = config.cfgimpl_get_settings()
    for path_var, properties in setting.get_modified_properties().items():
        if 'load_error' in properties:
            path = path_var.split('.')
            category_name = config.unwrap_from_path('.'.join(path[0:2])
                                                    ).impl_getdoc().capitalize()
            try:
                varname = config.unwrap_from_path('.'.join(path)).impl_getdoc()
                varid = path[-1]
                if isinstance(category_name, unicode):
                    category_name = unicode.encode(category_name, 'utf8')
                msg = "Problème d'import de la variable \"{1}\"({2}) de la catégorie \"{0}\"".format(category_name, varname, varid)
                load_errors[path_var] = msg
            except PropertiesOptionError:
                pass
    return load_errors

def convert_to_creole3(option, value):
    if option.impl_is_multi():
        if type(value) != list:
            # options "multi": on met la valeur dans une liste si besoin
            value = [value]
        vals = []
        for val in value:
            if val is None:
                vals.append(val)
            else:
                vals.append(convert_value(option, val))
        return vals
    else:
        if value is not None:
            return convert_value(option, value)
    return value

def convert_from_creole3(val):
    if isinstance(val, Multi):
        if val in (None, []):
            return ['']
        values = []
        for value in list(val):
            if value is None:
                values.append('')
            elif type(value) not in (unicode, str):
                values.append(str(value))
            else:
                values.append(value)
        return values
    else:
        if val is None:
            return ['']
        elif type(val) not in (unicode, str):
            return [str(val)]
        else:
            return [val]

def get_upgrade_infos(config):
    """renvoie les informations d'upgrade d'une configuration (version et erreurs de migration)
    """
    version_info = config.impl_get_information('upgrade', '')
    upgrade_infos = get_load_errors(config)
    return {'upgrade': version_info, 'load_errors': upgrade_infos}

def sort_dicts(dict1, dict2):
    name1 = os.path.basename(dict1[0])
    name2 = os.path.basename(dict2[0])
    if name1 == name2:
        # noms identiques: ordre des répertoires conservé
        return dict1[1].__cmp__(dict2[1])
    elif name1 > name2:
        return 1
    else:
        return -1

def is_package_dir(creoledir):
    """détecte si un répertoire de dictionnaires doit être lu en première
    passe par la librairie Creole3 (dictionnaires du module ou venant de paquets)
    """
    dirname, basename = os.path.split(creoledir)
    return (basename in ('module', 'package', 'var_package')) or \
           (basename == 'dicos' and os.path.dirname(dirname) == PATH_MODULES.rstrip('/'))

class ZephirDict:
    """sert de wrapper par dessus les dictionnaires creole1 et creole2
    """
    def __init__(self, dicos=[], confdir='', mode='config', version='creole2', \
                 force_instanciate='non', no_auto_store=False, \
                 eole_version=None, server_version=None):
        """initialise un objet dictionnaire avec la bonne api
        """
        self.version = version
        self.separators = {}
        self.multivars = []
        self.autovars = []
        self.confdir = confdir
        self.mode = mode
        self.index = 0
        self.warnings = False
        self.force_instanciate = force_instanciate
        self.no_auto_store = no_auto_store
        if eole_version in DISTRIBS:
            self.release_version = DISTRIBS[eole_version][1]
        else:
            self.release_version = None
        # release actuelle du serveur (utile si migration)
        self.server_version = server_version or eole_version
        # initialisiation du dictionnaire
        if self.version == "creole2":
            self.dico = EDict(logfile='NOLOG')
        elif self.version == "creole3":
            self.creoledirs = []
        elif self.version == "creole1":
            self.dico_zephir = None
            self.dico = None
        self.dicos = []
        # chargement de tous les dictionnaires disponibles
        if self.version == 'creole2':
            for creoledir in dicos:
                if os.path.isdir(creoledir):
                    # dictionnaires de module / package lus en premier
                    list_dic = glob(creoledir + '/*.xml')
                    list_dic.sort()
                    self.dicos.extend(list_dic)
                    self.dico.read_dir(creoledir)
            self._update_vars()
        elif self.version == 'creole3':
            start_list = []
            end_list = []
            for idx, creoledir in enumerate(dicos):
                if os.path.isdir(creoledir):
                    if is_package_dir(creoledir):
                        # dictionnaires de module / package lus en premier
                        for dict_file in glob(creoledir + '/*.xml'):
                            start_list.append((dict_file, idx))
                    else:
                        list_dic = glob(creoledir + '/*.xml')
                        list_dic.sort()
                        end_list.extend(list_dic)
            # tri de la liste de départ par nom de fichier / ordre
            start_list.sort(sort_dicts)
            self.dicos.extend([dict_info[0] for dict_info in start_list])
            self.dicos.extend(end_list)

        self.load_values(mode, dicos)

    def get_config_file(self, mode, write=False):
        """calcule le nom du fichier de configuration a utiliser en fonction du mode
        write : si True, indique qu'on cherche à sauvegarder le fichier
                si False, recherche le fichier à lire pour récupérer les valeurs déjà connues
        """
        config_file = ""
        if self.confdir != '':
            # le fichier doit exister
            # dans le cas de creole3, on crée un dictionnaire avec les valeurs sauvegardées
            # (chargements successifs de fichiers de valeurs non gérés)
            serv_file = os.path.join(self.confdir, 'zephir.eol')
            mod_file = os.path.join(self.confdir, 'module.eol')
            var_file = os.path.join(self.confdir, 'dico.eol')
            migration_file = os.path.join(self.confdir, 'migration.eol')
            if mode == 'modif_config' and os.path.isfile(serv_file):
                config_file = serv_file
            elif mode in ['config','modif_config','modif_dico']:
                if os.path.isfile(var_file):
                    config_file = var_file
                elif os.path.isfile(mod_file):
                    config_file = mod_file
            elif mode in ['migration', 'modif_migration'] and os.path.isfile(migration_file):
                config_file = migration_file
        return config_file

    def set_owner(self, owner):
        """spécifique creole3 : initialise le propriétaire d'une valeur
        """
        if self.dico is not None:
            if owner not in dir(owners):
                owners.addowner(owner)
            self.dico.cfgimpl_get_settings().setowner(getattr(owners, owner))

    def check_warnings(self):
        # Désactivation des warnings si besoin (désactivés par défaut)
        if self.warnings == False:
            settings = self.dico.cfgimpl_get_settings()
            if 'warnings' in settings:
                settings.remove('warnings')

    def load_dicos_creole3(self, dicos, from_zephir=False):
        # creole 3: on doit charger tous les dictionnaires en une passe
        self.dico = CreoleVarLoader(no_auto_store=self.no_auto_store)
        if self.release_version is None or self.release_version.startswith('2.4'):
            test_duplicate = False
        else:
            test_duplicate = True
        if from_zephir:
            self.creoledirs = []
            dicos = [base64.decodestring(dico_zeph) for dico_zeph in dicos[:-1]]
            # chargement de tous les dictionnaires disponibles
            self.dico.read_string(dicos, 'creole', test_duplicate=test_duplicate)
            self.groups = self.dico.groups
        elif dicos:
            creoledirs = []
            mod_dirs = []
            # regroupement des répertoire module / package / var_package
            for dict_dir in dicos:
                if is_package_dir(dict_dir):
                    mod_dirs.append(dict_dir)
                else:
                    creoledirs.append(dict_dir)
            if mod_dirs:
                creoledirs.insert(0, mod_dirs)
            self.creoledirs = creoledirs
            self.dico.read_dir(creoledirs, 'creole', force_test_duplicate=test_duplicate)
            # stockage des séparateurs
            self.separators = {}
            for sep, data in self.dico.separators.items():
                self.separators[sep] = (data[0], data[1] is not None)
            # stockage des groupes (master/slave)
            self.groups = self.dico.groups
        # après chargement des dictionnaires,
        # on ne conserve que la configuration tiramisu
        self.dico = self.dico.get_config()
        self.dico.read_write()
        # désactivation des propriétés 'hidden'
        # pour permettre l'accès aux variables cachées
        settings = self.dico.cfgimpl_get_settings()
        settings.remove('hidden')
        self.check_warnings()
        # par défaut, on utilise 'zephir' comme owner des valeurs modifiées
        # redéfini dans gen_config si édition au niveau variante ou module
        self.set_owner('zephir')

    def migrate_creole3(self, dicos):
        """fonction de migration d'une configuration creole2 vers creole3
        pour creole3, le fonctionnement est le suivant
        - stockage des valeurs par défaut de module/variante de la version cible dans un dictionnaire python
        - chargement de l'ancien fichier de configuration avec le loader creole (appel auto des fonctions de migration)
          et récupération des valeurs dans un dictionnaire
        - fusion des valeurs par défaut et des valeurs migrées
        - rechargement de la configuration avec ces valeurs
        """
        # fichier des valeurs actuelles
        current_eol = os.path.join(self.confdir, 'zephir.eol')
        migration_eol = os.path.join(self.confdir, 'migration.eol')
        # on essaye de charger les données de module.eol et dico.eol en premier
        def_values = {}
        for path_conf in dicos:
            for conf_file in ('module.eol','dico.eol'):
                fic_defaults = os.path.join(os.path.dirname(path_conf),conf_file)
                if os.path.isfile(fic_defaults):
                    try:
                        new_vals = cjson.decode(open(fic_defaults).read().strip(), all_unicode = True)
                        def_values.update(new_vals)
                    except:
                        pass
        if self.server_version < 6:
            # import depuis 2.3
            self.dico.impl_set_information('load_error', True)
            # migration des valeurs de la configuration actuelle
            serv_prefix = '-'.join(self.confdir.split(os.path.sep)[-2:])
            # modification du préfixe de log dans creole.log (ajout de RNE-N°SERV)
            upgrade.log.name = "creole3.upgrade.{0}".format(serv_prefix)
            migration_values, version = upgrade.upgrade(self.dico, unicode(current_eol, charset))
            def_values.update(migration_values)
            # on spécifie que la conf a été migrée en 2.4.0 pour permettre un upgrade jusqu'à la version finale
            def_values['eol_version'] = "2.4.0"
            try_upgrade = False
        else:
            version = self.release_version
            # chargement du fichier actuel
            store_values = load_store(self.dico, unicode(current_eol, charset))
            def_values.update(store_values)
            if 'eol_version' not in store_values:
                def_values['eol_version'] = version
            try_upgrade = True
        # application des migrations automatiques de valeurs
        settings = self.dico.cfgimpl_get_settings()
        settings.remove('disabled')
        settings.remove('frozen')
        load_config_store(self.dico,
                          def_values,
                          unset_default=True,
                          force_instanciate='non',
                          current_eol_version=version,
                          remove_unknown_vars=True,
                          try_upgrade=try_upgrade)
        self.dico.impl_set_information('upgrade', version)
        settings.append('frozen')
        settings.append('disabled')

    def _load_values_creole3(self, mode, dicos, config_file):
        if dicos:
            self.load_dicos_creole3(dicos)
        # if not config_file and not dicos and self.creoledirs:
        elif self.creoledirs:
            # création d'un nouvelle configuration depuis la description existante si
            # retour aux valeurs par défaut quand les dictionnaires sont déjà chargés
            cfg_descr = self.dico.cfgimpl_get_description()
            unknown_values = {}
            unknown_values.update(self.dico.impl_get_information('unknown_options', {}))
            self.dico = Config(cfg_descr)
            self.dico.impl_set_information('creole', unicode(config_file, charset))
            self.dico.impl_set_information('unknown_options', unknown_values)
            self.dico.read_write()
            settings = self.dico.cfgimpl_get_settings()
            settings.remove('hidden')
            _modes = list(modes_level)
            _modes.append('hidden')
            settings.setpermissive(tuple(_modes))
            self.check_warnings()
        # chargement du fichier de valeurs
        if mode == 'migration':
            self.migrate_creole3(dicos)
        elif config_file:
            store_values = {}
            config_files = [config_file]
            # cas particulier pour creole3. Dans le cas de chargement des valeurs par défaut
            # d'une variante, on charge aussi les valeurs au niveau module
            if os.path.basename(config_file) == 'dico.eol':
                config_files.insert(0, os.path.join(os.path.dirname(config_file), 'module.eol'))
            for conf_file in config_files:
                if os.path.isfile(conf_file):
                    try:
                        store_values.update(cjson.decode(open(conf_file).read().strip(), all_unicode = True))
                    except:
                        pass
            # on désactive les tests sur frozen pour pouvoir recharger les valeurs
            settings = self.dico.cfgimpl_get_settings()
            settings.remove('frozen')
            settings.remove('disabled')
            # si le serveur est déjà enregistré, on considère qu'il est instancié
            # (les variables auto_freeze ne sont plus modifiables)
            # valeurs passées sous forme de dictionnaires
            config_load_store(self.dico, 'creole', store_values, force_instanciate=self.force_instanciate)
            settings.append('disabled')
            settings.append('frozen')
        if hasattr(self, 'dico'):
            self._update_vars()

    def _load_values_creole2(self, mode, dicos, config_file):
        # nettoyage de l'objet store avant application des valeurs (ref #12679)
        self.dico.store = ConfigParser.ConfigParser(dict_type=dict)
        # reset des valeurs
        self.dico.reset_defaults()
        if config_file != "" and mode != 'migration':
            self.dico.load_values(config_file, check=False)
        elif mode in ['migration', 'modif_migration']:
            # on essaye de charger les données de module.eol et dico.eol en premier
            for path_conf in dicos:
                for conf_file in ('module.eol','dico.eol'):
                    if os.path.isfile(os.path.join(os.path.dirname(path_conf),conf_file)):
                        self.dico.load_values(os.path.join(os.path.dirname(path_conf),conf_file), check=False)
            # import des valeurs creole1 du serveur
            if os.path.isfile(self.confdir+os.sep+'zephir.eol'):
                # détection du format de l'ancien dictionnaire (creole 1 ou 2)
                first_line = open(self.confdir+os.sep+'zephir.eol').read().strip().split('\n')[0]
                if not (first_line.startswith('[') and first_line.endswith(']')):
                    # migration depuis une config creole1
                    res = self.dico.import_values_creole1(self.confdir+os.sep+'zephir.eol')
                else:
                    # récupération des valeurs migrées par upgrade.py
                    res = self.dico.import_values(self.confdir+os.sep+'zephir.eol')
        elif os.path.isfile(self.confdir+os.sep+'module.eol'):
            # on prend les valeurs par défaut du module
            self.dico.load_values(self.confdir+os.sep+'module.eol', check=False)

    def _load_values_creole1(self, mode, dicos, config_file):
        if dicos == [] and self.dico_zephir:
            dicos = self.dico_zephir
        # initialisiation du dictionnaire. Dans creole1, les dictionnaires et les valeurs sont dans
        # les même fichiers
        self.dico = dico.DicoEole(dico_zephir=dicos)
        # on conserve les dictionnaires pour pouvoir réinitialiser si chargement d'un autre fichier
        self.dico_zephir = dicos
        # dictionnaire des valeurs par défaut ou déjà remplies
        dico_actuel = None
        if config_file != "":
            fic = open(config_file)
            data = fic.readlines()
            lines = [ unicode(line,'ISO-8859-1').encode(charset) for line in data ]
            fic.close()
            # on parse ce dico
            dico_actuel = dico.DicoEole(dico_zephir=[lines]).ParseDico()

        if dico_actuel is not None:
            # des valeurs ont déjà été saisies
            dico_final = {}
            # on réinsère les données existantes dans le dictionnaire
            for variable, data in self.dico.dictionnaire.items():
                # si la variable existait déjà, on reprend sa valeur
                if dico_actuel.has_key(data[0]):
                    data[1] = dico_actuel[data[0]][0]
                dico_final[variable] = data
            # on remet le dictionnaire modifié en place dans self.dico
            self.dico.dictionnaire = dico_final
        self.liste_vars = self.dico.liste_vars

    def load_values(self, mode, dicos=[], config_file=None):
        """ lecture des valeurs actuelles selon le mode demandé
        """
        if config_file is None:
            config_file = self.get_config_file(mode)
        if self.version == 'creole3':
            self._load_values_creole3(mode, dicos, config_file)
        elif self.version != 'creole1':
            self._load_values_creole2(mode, dicos, config_file)
        else:
            self._load_values_creole1(mode, dicos, config_file)
        self.mode = mode

    def get_help(self, name, family=False):
        """Récupère l'aide de la variable
        - définie dans une balise help pour creole2 (chaine vide si non défini)
        - pour creole 1, on renvoie une chaine vide
        """
        if self.version == 'creole3':
            try:
                if family:
                    opt = self.dico.unwrap_from_path('creole.' + name)
                else:
                    opt = self.dico.vars[self.dico.liste_vars[name]]
            except:
                opt = None
            if opt: return opt.impl_getdoc()
        elif self.version == 'creole2':
            if family:
                section = 'families'
            else:
                section = 'variables'
            if name in self.dico.helps.get(section, {}):
                return self.dico.helps[section][name]
        return ""

    def init_from_zephir(self, dicos, load_error=False, force_load_owner=None, warnings=False, eole_version=None):
        """charge les dictionnaire depuis une liste de chaines xml
        @param dicos: contenu des dictionnaires à implémenter + valeurs actuelles en fin de liste
        @param load_error: passer à True pour creole3 dans le cas où la configuration doit être upgradée
        """
        # chargement depuis des valeurs envoyées, on ne sait pas dans quel mode on est
        self.mode = None
        self.warnings = warnings
        upgrade_infos = None
        # redéfinition du n° de release si passé en paramêtre
        if eole_version in DISTRIBS:
            self.release_version = DISTRIBS[eole_version][1]
        if self.version != 'creole1':
            if self.dicos == []:
                # Les dictionnaires n'ont pas été chargés précédemment, on utilise ceux fournis
                if self.version == 'creole3':
                    if type(dicos[-1] == dict) and 'load_errors' in dicos[-1]:
                        upgrade_infos = dicos[-1]
                        dicos = dicos[:-1]
                    self.load_dicos_creole3(dicos, True)
                else:
                    self.dico = EDict(logfile='NOLOG')
                    self.dicos = [base64.decodestring(dico_zeph) for dico_zeph in dicos[:-1]]
                    # chargement de tous les dictionnaires disponibles
                    for xml_dict in self.dicos:
                        self.dico.read_string(xml_dict)
            # prise en compte du dictionnaire de valeurs
            if self.version == "creole3":
                values = eval(dicos[-1])
                #on reinitialise aux valeurs par defaut
                cfg_descr = self.dico.cfgimpl_get_description()
                self.dico = Config(cfg_descr)
                _modes = list(modes_level)
                _modes.append('hidden')
                settings = self.dico.cfgimpl_get_settings()
                settings.setpermissive(tuple(_modes))
                if not load_error:
                    config_load_store(self.dico, 'creole', values,
                                      force_instanciate=self.force_instanciate,
                                      force_load_owner=force_load_owner)
                    self.dico.impl_set_information('orig_values', values)
                    if upgrade_infos:
                        self.dico.impl_set_information('upgrade', upgrade_infos['upgrade'])
                        # réinjection des erreurs dans les variables
                        for opt_path, error in upgrade_infos['load_errors'].items():
                            opt = self.dico.unwrap_from_path(opt_path)
                            settings[opt].append('load_error')
                            path_split = opt_path.split('.')
                            family_option = self.dico.unwrap_from_path(path_split[0] + '.' + path_split[1])
                            settings.setpermissive(tuple(modes_level), opt=family_option)
                            if len(path_split) == 4:
                                parent_option = self.dico.unwrap_from_path(path_split[0] + '.' + path_split[1] + '.' + path_split[2])
                                settings.setpermissive(tuple(modes_level), opt=parent_option)
                            settings.setpermissive(tuple(modes_level), opt=opt)
                else:
                    self.dico.impl_set_information('load_error', True)
                    # Try to upgrade
                    serv_prefix = '-'.join(self.confdir.split(os.path.sep)[-2:])
                    # modification du préfixe de log dans creole.log (ajout de RNE-N°SERV)
                    upgrade.log.name = "creole3.upgrade.{0}".format(serv_prefix)
                    try:
                        # upgrade fonctionne seulement en mode fichier, on écrit un fichier temporaire
                        check_path_temp()
                        descr, upg_filename = mkstemp(suffix="{0}_upgrade.eol".format(serv_prefix), dir=PATH_TEMP)
                        f_upgr = open(upg_filename, 'w')
                        parser = ConfigParser.ConfigParser(dict_type=dict)
                        if isinstance(values, OrderedDict):
                            parser._sections = dict(values)
                        else:
                            parser._sections = values
                        parser.write(f_upgr)
                        f_upgr.close()
                        # migration des valeurs et détection de la version d'origine
                        store_dico, version = upgrade.upgrade(self.dico, upg_filename)
                        config_load_store(self.dico, 'creole', store_dico, unset_default=True,
                                         force_instanciate=self.force_instanciate,
                                         force_load_owner=force_load_owner)
                        self.dico.impl_set_information('upgrade', version)
                        self.dico.impl_set_information('orig_values', store_dico)
                        # suppression du fichier temporaire après upgrade
                        close(descr)
                        os.unlink(upg_filename)
                    except Exception, e:
                        traceback.print_exc()
                        upgrade.log.error("fichier de configuration invalide (eole 1.2 ou 2.3)")
                self.dico.read_write()
                settings.remove('hidden')
                self.check_warnings()
            else:
                # chargement des valeurs actuelles
                val_dict = eval(dicos[-1])
                if isinstance(val_dict, OrderedDict):
                    self.dico.store._sections = dict(val_dict)
                else:
                    self.dico.store._sections = val_dict
                self.dico._load_values()
            if len(dicos) > 1:
                # si dictionnaires réinitialisés,
                # on réinitialise les données de structure
                self._update_vars()
        else:
            dicos = base64.decodestring(dicos)
            dicos = [dicos.splitlines(True)]
            self.dico = dico.DicoEole(dico_zephir=dicos)
            self.liste_vars = self.dico.liste_vars

    def _update_vars(self):
        """regénère les structures après chargement des dictionnaires
        """
        if self.version != 'creole1':
            # création d'un liste pour ordonner les variables
            self.index = 0
            self.vars = []
            # création de self.liste_vars
            self.liste_vars = {}
            ind = 0
            if self.version == 'creole3':
                # dictionnaire pour retrouver le chemin d'une variable donnée
                self.var_paths = {}
                # liste des familles ordonnée
                self.families = []
                self.expert_families = []
                # lecture du cache ordonné des variables
                # on n'a pas besoin de conserver la liste des variables multi/auto
                descr_cache = self.dico.cfgimpl_get_description()._cache_paths
                for cache_index, descr in enumerate(descr_cache[0]):
                    if isinstance(descr, OptionDescription):
                        if getattr(descr, '_group_type', '') == 'family':
                            current_mode = 'normal'
                            # vérification du mode expert au niveau de la famille
                            if 'expert' in descr._properties:
                                self.expert_families.append(descr)
                                current_mode = 'expert'
                            else:
                                self.families.append(descr)
                    else:
                        current_path = descr_cache[1][cache_index]
                        self.vars.append(descr)
                        self.liste_vars[descr._name] = ind
                        self.var_paths[descr._name] = (current_path, current_mode)
                        ind += 1
            else:
                # creole 2
                for family in self.dico.family_order:
                    for var in self.dico.families[family]['vars']:
                        self.vars.append(var)
                        self.liste_vars[var] = ind
                        ind += 1
                        if self.dico.variables[var].multi == True:
                            self.multivars.append(var)
                            self.multivars.extend(self.dico.groups.get(var,[]))
                        if self.dico.variables[var].auto != []:
                            self.autovars.append(var)
                # séparateurs (sections dans un onglet)
                self.separators = self.dico.separators

    def get_value(self, var):
        if self.version == 'creole3':
            # Si la variable est 'disabled', on ne peut pas accéder à la valeur
            try:
                val = getattr(self.dico, self.var_paths[var][0]) or ''
            except PropertiesOptionError, e:
                val = ''
            return convert_from_creole3(val)
        else:
            return self.dico.get_value(var)

    def get_enumeration(self, var):
        vals = []
        allow_other = False
        if self.version == 'creole3':
            try:
                option = self.vars[self.liste_vars[var]]
            except AttributeError, e:
                option = None
            if option and isinstance(option, ChoiceOption):
                vals = list(option.impl_get_values())
                allow_other = option.impl_is_openvalues()
        elif self.version != 'creole1':
            eolvar = self.dico.get_variable(var)
            checks = eolvar.checks
            for check in checks:
                if check[0] == u'valid_enum':
                    # récupération des valeurs possibles
                    for param in check[1]:
                        if param['name'] == 'checkval':
                            if param['value'] == "False":
                                allow_other = True
                        else:
                            vals.extend(eval(param['value']))
        else:
            conditions = self.dico.dictionnaire[self.dico.liste_vars[var]][3]
            # recherche des enumérations
            for cond in conditions:
                if cond.startswith('enumeration'):
                    try:
                        vals = eval(cond.split('??')[1])
                    except:
                        # si rien n'est spécifié, on propose oui,non par défaut
                        vals = ['oui','non']
                    break
        return vals, allow_other

    def get_first(self):
        """retourne les paramètres de la variable nom,valeur,libelle,variable cachée
        """
        if self.version == 'creole1':
            self.dico.get_first()
        else:
            self.index = 0
        try:
            return self._get_var_details()
        except PropertiesOptionError, e:
            # pour creole3, on gère le cas ou la première
            # variable serait désactivée
            return self.get_next()

    def get_next(self):
        if self.version == 'creole1':
            self.dico.get_next()
        else:
            prev_index = self.index
            self.index = self.index + 1
        try:
            return self._get_var_details()
        except PropertiesOptionError, e:
            # variable non accessible, on passe à la suivante
            return self.get_next()
        except (KeyError, IndexError), e:
            self.index = prev_index
            raise dico.PositionError, 'fin de la liste des variables'

    def get_var(self, varname=None, default=False):
        """retourne les paramètres d'une variable
        """
        try:
            return self._get_var_details(varname, default)
        except PropertiesOptionError, e:
            # FIXME variable disabled, à gérer différement ?
            return ['']

    def _get_var_details(self, varname=None, default=False):
        """adapte la représentation des variables
        """
        # récupération des infos
        if self.version == 'creole3':
            # creole3 : on  ne récupère pas toutes les informations
            # (nom et description seulement). Le formulaire de saisie
            # est géré indépendament
            if varname != None:
                self.index = self.liste_vars[varname]
            opt = self.vars[self.index]
            # adaptation du mode
            if 'expert' in opt._properties:
                mode = 'expert'
            else:
                # pas de distinction entre basic et normal
                mode = 'normal'
            hidden = check_hidden(opt._properties)
            obligatoire = 'mandatory' in opt._properties
            mime = opt._opt_type
            val = getattr(self.dico, self.var_paths[opt._name][0])
            if not val:
                if isinstance(val, Multi):
                    val = ['']
                else:
                    val = ''
            data = opt._name, val, opt.impl_getdoc(), hidden, obligatoire, mime, mode
        elif self.version == 'creole2':
            if varname != None:
                var = self.dico.get_variable(varname)
                self.index = self.vars.index(varname)
            else:
                var = self.dico.get_variable(self.vars[self.index])
            obligatoire = False
            if 'obligatoire' in [ check[0] for check in var.checks ]:
                obligatoire = True
            mime = var.mime
            mode = var.mode
            val = var.get_value(default=default)
            if type(val) != list:
                val = [val]
            if val == []:
                val = ['']
            data = var.name, val, var.description, var.hidden, obligatoire, mime, mode
        elif self.version == 'creole1':
            if varname != None:
                name, val, description, hidden = self.dico.get_var(varname)
            else:
                name, val, description, hidden = self.dico.get_current()
            checks = self.dico.dictionnaire[self.dico.liste_vars[name]][3]
            if 'obligatoire' in checks:
                obligatoire = True
                liste_funcs = copy(checks)
                liste_funcs.remove('obligatoire')
                # liste_funcs = checks.remove('obligatoire')
            else:
                obligatoire = False
                liste_funcs = checks
            # pas de mode expert au niveau variable dans Creole 1
            mode = 'normal'
            data = name, [val], description, hidden, obligatoire, liste_funcs, mode
        return data

    def get_prec_value(self, varname):
        """ renvoie la valeur précédente """
        if self.version != 'creole1':
            try:
                value = self.dico.get_prec_value(varname)
            except:
                value = ''
        else:
            try:
                value = self.dico.get_var(varname)[1]
            except:
                value = ''
        return value

    def get_default_value(self, varname):
        """ renvoie la valeur par défaut """
        try:
            if self.version == 'creole3':
                value = self.vars[self.liste_vars[varname]].impl_getdefault()
            else:
                value = self.dico.get_var(varname, default=True)[1]
        except:
            value = ''
        return value

    def set_value(self, value, invisible=False, force=False):
        if self.version == 'creole3':
            try:
                option = self.vars[self.index]
                varname = option._name
                # reprise du fonctionnement de eole-genconfig (conversion + mode permissive)
                value = convert_to_creole3(option, value)
                self.dico.cfgimpl_get_settings().setpermissive(tuple(modes_level), opt=option)
                setattr(self.dico, self.var_paths[varname][0], value)
                # rechargement des familles/variables accessibles
                self._update_vars()
            except Exception, e:
                traceback.print_exc()
                raise dico.InvalidDataError, e.args[0]
        elif self.version != 'creole1':
            try:
                self.dico.set_value(self.vars[self.index], value, force=force)
            except Exception, e:
                raise dico.InvalidDataError, e.args[0]
        else:
            if type(value) == list:
                value = value[0]
            self.dico.set_value(value, invisible, force=force)

    def save(self, eol_file='/dev/null', force=False, encode=False):
        if self.version == 'creole3':
            # vérifier eole_file : config.impl_get_information(namespace)
            check_mandatory=not force
            if eol_file and eol_file != '/dev/null':
                # Vérification de la destination d'écriture
                if not (eol_file.startswith(PATH_ZEPHIR) or eol_file.startswith(PATH_TEMP)):
                    print "ERR : écriture dans {0}".format(eol_file)
                    raise IOError("Destination invalide pour le fichier de configuration: {0}".format(eol_file))
                if eol_file.startswith(PATH_TEMP):
                    check_path_temp()
                config_save_values(self.dico, 'creole', eol_file=eol_file, check_mandatory=check_mandatory, eol_version=self.release_version)
            data = [str(config_get_values(self.dico, 'creole', check_mandatory))]
        elif self.version != 'creole1':
            # sauvegarde du fichier de valeurs
            self.dico.save_values(eol_file,force=force)
            # on retourne le dictionnaire des valeurs pour recréer le ConfigParser
            if isinstance(self.dico.store._sections, OrderedDict):
                data = [str(dict(self.dico.store._sections))]
            else:
                data = [str(self.dico.store._sections)]
        else:
            data = self.dico.save(eol_file, encode=encode)
        return data

    def get_dict(self):
        # on envoie le dictionnaire sous forme de fichier texte
        data = []
        if self.version != 'creole1':
            for dic in self.dicos:
                f = open(dic)
                content = f.read()
                f.close()
                data.append(base64.encodestring(content))
            if self.version == 'creole3':
                val_store = config_get_values(self.dico, 'creole', check_mandatory=False, ignore_autofreeze=True)
                data.append(str(val_store))
            else:
                # suite à correction du bug #1280 , on assigne manuellement les valeurs
                # dans le ConfigParser du dictionnaire
                for varname in self.dico.variables.keys():
                    var = self.dico.variables[varname]
                    if var.val == [] or var.val == ['']:
                        # la variable est hidden (ou une famille la contenant)
                        # on prend la valeur par défaut si pas de valeur
                        var.val = var.get_value(default=True)
                    # passage des chaines unicode en str
                    for list_val in [var.val, var.valprec, var.valdefault]:
                        for ind in range(len(list_val)):
                            if type(list_val[ind]) == unicode:
                                list_val[ind] = list_val[ind].encode(charset)
                    # assignation des valeurs en interne dans le dictionnaire
                    if not self.dico.store.has_section(varname):
                        self.dico.store.add_section(varname)
                    self.dico.store.set(varname, 'val', var.val)
                    self.dico.store.set(varname, 'valprec', var.valprec)
                    self.dico.store.set(varname, 'valdefault', var.valdefault)
                if isinstance(self.dico.store._sections, OrderedDict):
                    data.append(str(dict(self.dico.store._sections)))
                else:
                    data.append(str(self.dico.store._sections))
            return data
        else:
            return [base64.encodestring(self.save())]

    def get_menu(self, expert=False):
        """retourne les familles et les variables dans l'ordre de saisie
        @param expert: mode de saisie (non expert par défaut)
        @return: renvoie une liste ordonnée de familles [famille, liste_vars]
        """
        families = []
        expert_families = []
        menu = []
        if self.version != 'creole1':
            if self.version == 'creole3':
                for family in self.families:
                    childs = family._children[0]
                    hidden = check_hidden(self.dico.unwrap_from_path('creole.%s' % family._name)._properties)
                    families.append([family._name.encode(charset), hidden, childs])
                if expert:
                    for family in self.expert_families:
                        childs = family._children[0]
                        hidden = check_hidden(self.dico.unwrap_from_path('creole.%s' % family._name)._properties)
                        expert_families.append([family._name.encode(charset), hidden, childs])
            else:
                # liste des familles normales
                for family in self.dico.family_order:
                    data = self.dico.families[family]
                    if not data['mode'] == 'expert':
                        # on récupère la liste des variables à afficher dans l'ordre
                        families.append([family.encode(charset), data['hidden'], data['vars']])
                    # liste des familles expert
                    elif expert == True:
                        expert_families.append([family.encode(charset), data['hidden'], data['vars']])
            menu = families + expert_families
        else:
            # on met toutes les variables (dans l'ordre de saisie) dans une famille "general".
            middle = len(self.dico.dictionnaire)/2
            end = len(self.dico.dictionnaire)-middle
            vars1 = [self.dico.dictionnaire[numvar][0] for numvar in range(middle)]
            vars2 = [self.dico.dictionnaire[middle + numvar][0] for numvar in range(end)]
            menu.append(['general', False, vars1])
            menu.append(['general_1', False, vars2])
        return menu

    def parsedico(self, separator=', '):
        # renvoie la configuration sous forme d'un dictionnaire {variable:valeur}
        data = {}
        if self.version == 'creole3':
            # passage en mode lecture pour activer les 'settings de base'
            data_dict = self.dico.make_dict(flatten=True)
            # mise en place des séparateurs
            for varname, vals in data_dict.items():
                vals = convert_from_creole3(vals)
                data[varname] = separator.join(vals)
        elif self.version != 'creole1':
            for varname in self.dico.variables.keys():
                data[varname] = separator.join(self.dico.get_value(varname))
        else:
            for varname, values in self.dico.ParseDico().items():
                data[varname] = values[0]
        return data
