//  ************************************************************************************************
//
//  BornAgain: simulate and fit reflection and scattering
//
//! @file      GUI/Model/Descriptor/DoubleProperty.h
//! @brief     Defines class DoubleProperty
//!
//! @homepage  http://www.bornagainproject.org
//! @license   GNU General Public License v3 or higher (see COPYING)
//! @copyright Forschungszentrum Jülich GmbH 2021
//! @authors   Scientific Computing Group at MLZ (see CITATION, AUTHORS)
//
//  ************************************************************************************************

#ifndef BORNAGAIN_GUI_MODEL_DESCRIPTOR_DOUBLEPROPERTY_H
#define BORNAGAIN_GUI_MODEL_DESCRIPTOR_DOUBLEPROPERTY_H

#include "Fit/Param/RealLimits.h"
#include "GUI/Support/Type/Unit.h"
#include <QList>
#include <QString>
#include <QXmlStreamReader>
#include <variant>

//! Class for representing a double value, its attributes and its accessors.
//!
//! Contained attributes are
//! * the value itself
//! * label: a label of e.g. a spin box
//! * tooltip: tooltip for e.g. a spin box
//! * unit: unit of this value (not the representation on UI, but the unit of the contained value).
//!   Use this to show the unit in the UI as well as allow a representation in a different unit
//!   (e.g. Angstrom instead of nanometer)
//! * decimals: how many decimals shall be shown in a spin box
//! * limits: which limits shall be set in a spin box or other validator
//! * uid: a unique id which represents this property. This is used for linking to this property,
//!   also when serializing the link. Right now, this uid is a UUID.

class DoubleProperty {
public:
    void init(const QString& label, const QString& tooltip, double value,
              const std::variant<QString, Unit>& unit, const QString& uidPrefix);
    void init(const QString& label, const QString& tooltip, double value,
              const std::variant<QString, Unit>& unit, uint decimals, const RealLimits& limits,
              const QString& uidPrefix);
    void init(const QString& label, const QString& tooltip, double value,
              const std::variant<QString, Unit>& unit, uint decimals, double step,
              const RealLimits& limits, const QString& uidPrefix);

    //! Cast to the contained double value.
    operator double() const { return m_value; }

    double value() const { return m_value; }
    void setValue(double d) { m_value = d; }

    QString uid() const { return m_uid; }
    void setUid(const QString& uid) { m_uid = uid; }

    double step() const { return m_step; }

    QString label() const { return m_label; }

    uint decimals() const { return m_decimals; }
    void setDecimals(uint decimals) { m_decimals = decimals; }

    QString tooltip() const { return m_tooltip; }
    void setTooltip(const QString& tooltip);

    std::variant<QString, Unit> unit() const { return m_unit; }
    void setUnit(const std::variant<QString, Unit>& unit);

    RealLimits limits() const { return m_limits; }
    void setLimits(const RealLimits& limits);

    //! True if one of the init methods has been called (checks for a valid uid).
    bool isInitialized() const;

    void writeTo(QXmlStreamWriter* w) const;
    void writeTo(QXmlStreamWriter* w, const QString& tag) const;
    void readFrom(QXmlStreamReader* r);
    void readFrom(QXmlStreamReader* r, const QString& tag);

private:
    double m_value = 0.0; //!< Current value
    QString m_uid;        //!< Unique id of this double property.

    // without serialization
    QString m_label;      //!< A label text (short, no trailing colon)
    QString m_tooltip;    //!< Tooltip text
    uint m_decimals = 3;  //!< numbers of decimals to be shown in an edit control
    double m_step = 0.01; //!< numbers of decimals to be shown in an edit control
    RealLimits m_limits;  //!< Limits of the value.
    std::variant<QString, Unit> m_unit =
        Unit::unitless; //!< Unit of the value (internal unit only!)
};

using DoubleProperties = QList<DoubleProperty*>;

#endif // BORNAGAIN_GUI_MODEL_DESCRIPTOR_DOUBLEPROPERTY_H
