///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file ModifierApplication.h
 * \brief Contains the definition of the Core::ModifierApplication class.
 */

#ifndef __OVITO_MODIFIER_APPLICATION_H
#define __OVITO_MODIFIER_APPLICATION_H

#include <core/Core.h>
#include "Modifier.h"

namespace Core {

class ModifiedObject;	// defined in ModifiedObject.h


/**
 * \brief Stores information about a particlar application of a Modifier
 *        instance in a geometry pipeline.
 *
 * \author Alexander Stukowski
 * \sa Modifier, ModifiedObject
 * \sa Modifier::modifierApplications()
 */
class CORE_DLLEXPORT ModifierApplication : public RefTarget
{
private:

	/// \brief Deserialization constructor.
	/// \param isLoading Indicates whether the object is being loaded from a file.
	///                  This parameter is only used by the object serialization system.
	ModifierApplication(bool isLoading = false);

public:

	/// \brief Constructs an application object for a given Modifier instance.
	/// \param modifier The modifier that is going to be applied in a geometry pipeline.
	ModifierApplication(Modifier* modifier);

	/// \brief Returns the modifier instance that is applied in a particalur geometry pipeline.
	/// \return The modifier instance.
	Modifier* modifier() const { return _modifier; }

	/// \brief Returns the geometry pipeline in which the modifier is used.
	/// \return The ModifiedObject this application is used in.
	ModifiedObject* modifiedObject() const;

	/// \brief Return the status returned by the modifier during its last evaluation.
	const EvaluationStatus& status() const { return _evalStatus; }

	/// \brief Stores the status of this modifier application.
	///
	/// \note This is an internal function.
	void setStatus(const EvaluationStatus& status);

private:

	/// The modifier that is being applied.
	ReferenceField<Modifier> _modifier;

	/// The status returned by the modifier during its last evaluation.
	EvaluationStatus _evalStatus;

	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(ModifierApplication)
	DECLARE_REFERENCE_FIELD(_modifier)

	friend class ModifiedObject;
};


};

#endif // __OVITO_MODIFIER_APPLICATION_H
